#' @name deleteArms
#' @title Delete Arms From a Project
#' 
#' @description Delete arms from a project.  This is a destructive action that
#'   will result in deletion of any events and data associated with the arm.
#'   Due to its destructive nature, it may only be performed on databases 
#'   in development status.  Data loss is non-reversible.
#'   
#' @param rcon A REDCap connection object as generated by 
#'   \code{redcapConnection}.
#' @param arms \code{character}, a vector of arm numbers that will be 
#'   deleted. May also be \code{integerish}.
#' @param refresh \code{logical(1)} If \code{TRUE}, the cached arms data will
#'   be refreshed after the deletion.
#' @param ... Additional arguments to pass to other methods.
#' @param error_handling An option for how to handle errors returned by the API.
#'   see \code{\link{redcap_error}}
#' @param config \code{list} Additional configuration parameters to pass to 
#'   \code{\link[httr]{POST}}. These are appended to any parameters in 
#'   \code{rcon$config}.
#' @param api_param \code{list} Additional API parameters to pass into the
#'   body of the API call. This provides users to execute calls with options
#'   that may not otherwise be supported by \code{redcapAPI}.
#' 
#' @section REDCap API Documentation:
#' This method allows you to delete Arms from a project. 
#' Notice: Because of this method's destructive nature, it is only 
#' available for use for projects in Development status. Additionally, 
#' please be aware that deleting an arm also automatically deletes all 
#' events that belong to that arm, and will also automatically delete 
#' any records/data that have been collected under that arm (this is 
#' non-reversible data loss).
#' 
#' NOTE: This only works for longitudinal projects.
#' 
#' @section REDCap Version:
#' At least 8.1.17+ (and likely some earlier versions)
#' 
#' @return 
#' None.
#' 
#' @references
#' Please refer to your institution's API documentation.
#' 
#' @export

deleteArms <- function(rcon, 
                       arms, 
                       ...){
  UseMethod("deleteArms")
}


#' @rdname deleteArms
#' @export

deleteArms.redcapApiConnection <- function(rcon, 
                                           arms, 
                                           refresh        = TRUE,
                                           ...,
                                           error_handling = getOption("redcap_error_handling"), 
                                           config         = list(), 
                                           api_param      = list()){
  
  if (is.numeric(arms)) arms <- as.character(arms)

   ##################################################################
  # Argument Validation
  
  coll <- checkmate::makeAssertCollection()
  
  checkmate::assert_character(arms, 
                              any.missing = FALSE,
                              add = coll)
  
  checkmate::assert_logical(x = refresh, 
                            len = 1, 
                            add = coll)
  
  error_handling <- checkmate::matchArg(x = error_handling, 
                                        choices = c("null", "error"), 
                                        .var.name = "error_handling",
                                        add = coll)
  
  checkmate::assert_list(x = config, 
                         names = "named", 
                         add = coll)
  
  checkmate::assert_list(x = api_param, 
                         names = "named", 
                         add = coll)
  
  checkmate::reportAssertions(coll)
  
  Arms <- rcon$arms()
  
  checkmate::assert_subset(x = arms, 
                           choices = as.character(Arms$arm_num), 
                           add = coll)
  
  checkmate::reportAssertions(coll)
  
  ###################################################################
  # Make API Body List
  body <- c(list(token = rcon$token,
                 content = "arm",
                 action = "delete"),
            vectorToApiBodyList(arms, "arms"))

  body <- body[lengths(body) > 0]
  
  ###################################################################
  # Call the API

  if (length(arms) > 0){ # Skip the call if there are no arms to delete
    response <- makeApiCall(rcon, 
                            body = c(body, api_param), 
                            config = config)
    
    if (response$status_code != 200) return(redcap_error(response, error_handling))
  }
  
  message(sprintf("Arms Deleted: %s", 
                  if (length(arms) > 0) paste0(arms, collapse = ", ") else "None."))
  
  if (refresh && rcon$has_arms()){
    rcon$refresh_arms()
    rcon$refresh_projectInformation()
  }
}
