% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/diagnose.R
\name{sampling_distribution}
\alias{sampling_distribution}
\title{Simulate the sampling distribution of estimates from a population}
\usage{
sampling_distribution(fit, data, fn = tidy, nsim = 100, fixed_x = TRUE, ...)
}
\arguments{
\item{fit}{A model fit to data, such as by \code{lm()} or \code{glm()}, to refit to
each sample from the population.}

\item{data}{Data drawn from a \code{population()}, using \code{sample_x()} and possibly
\code{sample_y()}. The \code{population()} specification is used to draw the samples.}

\item{fn}{Function to call on each new model fit to produce a data frame of
estimates. Defaults to \code{broom::tidy()}, which produces a tidy data frame of
coefficients, estimates, standard errors, and hypothesis tests.}

\item{nsim}{Number of simulations to run.}

\item{fixed_x}{If \code{TRUE}, the default, the predictor variables are held fixed
and only the response variables are redrawn from the population. If
\code{FALSE}, the predictor and response variables are drawn jointly.}

\item{...}{Additional arguments passed to \code{fn} each time it is called.}
}
\value{
Data frame (tibble) of \code{nsim + 1} simulation results, formed by
concatenating together the data frames returned by \code{fn}. The \code{.sample}
column identifies which simulated sample each row came from. Rows with
\code{.sample == 0} come from the original \code{fit}.
}
\description{
Repeatedly refits the model to new samples from the population, calculates
estimates for each fit, and compiles a data frame of the results.
}
\details{
To generate sampling distributions of different quantities, the user can
provide a custom \code{fn}. The \code{fn} should take a model fit as its argument and
return a data frame. For instance, the data frame might contain one row per
estimated coefficient and include the coefficient and its standard error; or
it might contain only one row of model summary statistics.

Refitting is done using the S3 generic \code{update()}, so this function can be
used with any model fit that supports \code{update()}. In base R, this includes
\code{lm()} and \code{glm()}, and many other model fits.
}
\examples{
pop <- population(
  x1 = predictor("rnorm", mean = 4, sd = 10),
  x2 = predictor("runif", min = 0, max = 10),
  y = response(0.7 + 2.2 * x1 - 0.2 * x2, error_scale = 4.0)
)

d <- sample_x(pop, n = 20) |>
  sample_y()

fit <- lm(y ~ x1 + x2, data = d)
# using the default fn = broom::tidy(). conf.int argument is passed to
# broom::tidy()
samples <- sampling_distribution(fit, d, conf.int = TRUE)
samples

suppressMessages(library(dplyr))
# the model is correctly specified, so the estimates are unbiased:
samples |>
  group_by(term) |>
  summarize(mean = mean(estimate),
            sd = sd(estimate))

# instead of coefficients, get the sampling distribution of R^2
rsquared <- function(fit) {
  data.frame(r2 = summary(fit)$r.squared)
}
sampling_distribution(fit, d, rsquared, nsim = 10)
}
\seealso{
\code{\link[=parametric_boot_distribution]{parametric_boot_distribution()}} to simulate draws from a fitted
model, rather than from the population
}
