% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/residuals.R
\name{binned_residuals}
\alias{binned_residuals}
\title{Obtain binned residuals for a model}
\usage{
binned_residuals(fit, predictors = !".fitted", breaks = NULL, ...)
}
\arguments{
\item{fit}{The model to obtain residuals for. This can be a model fit with
\code{lm()} or \code{glm()}, or any model that has \code{residuals()} and \code{fitted()}
methods.}

\item{predictors}{Predictors to calculate binned residuals for. Defaults to
all predictors, skipping factors. Predictors can be specified using
tidyselect syntax; see \code{help("language", package = "tidyselect")} and the
examples below. Specify \code{predictors = .fitted} to obtain binned residuals
versus fitted values.}

\item{breaks}{Number of bins to create. If \code{NULL}, a default number of breaks
is chosen based on the number of rows in the data.}

\item{...}{Additional arguments passed on to \code{residuals()}. The most useful
additional argument is typically \code{type}, to select the type of residuals to
produce (such as standardized residuals or deviance residuals).}
}
\value{
Data frame (tibble) with one row per bin \emph{per selected predictor},
and the following columns:

\item{.bin}{Bin number.}
\item{n}{Number of observations in this bin.}
\item{predictor_name}{Name of the predictor that has been binned.}
\item{predictor_min, predictor_max, predictor_mean, predictor_sd}{Minimum,
maximum, mean, and standard deviation of the predictor (or fitted values).}
\item{resid_mean}{Mean residual in this bin.}
\item{resid_sd}{Standard deviation of residuals in this bin.}
}
\description{
Construct a data frame by binning the fitted values or predictors of a model
into discrete bins of equal width, and calculating the average value of the
residuals within each bin.
}
\details{
In many generalized linear models, the residual plots (Pearson or deviance)
are not useful because the response variable takes on very few possible
values, causing strange patterns in the residuals. For instance, in logistic
regression, plotting the residuals versus covariates usually produces two
curved lines.

If we first bin the data, i.e. divide up the observations into \code{breaks} bins
based on their fitted values, we can calculate the average residual within
each bin. This can be more informative: if a region has 20 observations and
its average residual value is large, this suggests those observations are
collectively poorly fit. We can also bin each predictor and calculate
averages within those bins, allowing the detection of misspecification for
specific model terms.
}
\section{Limitations}{
Factor predictors (as factors, logical, or character vectors) are detected
automatically and omitted. However, if a numeric variable is converted to
factor in the model formula, such as with \code{y ~ factor(x)}, the function
cannot determine the appropriate type and will raise an error. Create factors
as needed in the source data frame \emph{before} fitting the model to avoid this
issue.
}

\examples{
fit <- lm(mpg ~ disp + hp, data = mtcars)

# Automatically bins both predictors:
binned_residuals(fit, breaks = 5)

# Just bin one predictor, selected with tidyselect syntax. Multiple could be
# selected with c().
binned_residuals(fit, disp, breaks = 5)

# Bin the fitted values:
binned_residuals(fit, predictors = .fitted)

# Bins are made using the predictor, not regressors derived from it, so here
# disp is binned, not its polynomial
fit2 <- lm(mpg ~ poly(disp, 2), data = mtcars)
binned_residuals(fit2)
}
\references{
Gelman, A., Hill, J., and Vehtari, A. (2021). \emph{Regression and
Other Stories}. Section 14.5. Cambridge University Press.
}
\seealso{
\code{\link[=partial_residuals]{partial_residuals()}} for the related partial residuals;
\code{vignette("logistic-regression-diagnostics")} and
\code{vignette("other-glm-diagnostics")} for examples of use and interpretation
of binned residuals in logistic regression and GLMs; \code{\link[=bin_by_interval]{bin_by_interval()}}
and \code{\link[=bin_by_quantile]{bin_by_quantile()}} to bin data and calculate other values in each bin
}
