% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Random_Numbers.R
\name{rtelliptical}
\alias{rtelliptical}
\title{Sampling Random Numbers from Truncated Multivariate Elliptical Distributions}
\usage{
rtelliptical(n = 10000, mu = rep(0, length(lower)),
  Sigma = diag(length(lower)), lower, upper = rep(Inf, length(lower)),
  dist = "Normal", nu = NULL, expr = NULL, gFun = NULL,
  ginvFun = NULL, burn.in = 0, thinning = 1)
}
\arguments{
\item{n}{number of observations to generate. Must be an integer >= 1.}

\item{mu}{numeric vector of length \eqn{p} representing the location parameter.}

\item{Sigma}{numeric positive definite matrix with dimension \eqn{p}x\eqn{p} representing the
scale parameter.}

\item{lower}{vector of lower truncation points of length \eqn{p}.}

\item{upper}{vector of upper truncation points of length \eqn{p}.}

\item{dist}{represents the truncated distribution to be used. The values are \code{'Normal'},
\code{'t'}, \code{'Laplace'}, \code{'PE'}, \code{'PVII'}, \code{'Slash'}, and \code{'CN'} for the truncated Normal, Student-t,
Laplace, Power Exponential, Pearson VII, Slash, and Contaminated Normal distribution, respectively.}

\item{nu}{additional parameter or vector of parameters depending on the
density generating function. See Details.}

\item{expr}{a character with the density generating function. See Details.}

\item{gFun}{an R function with the density generating function. See Details.}

\item{ginvFun}{an R function with the inverse of the density generating function defined in
\code{gFun}. See Details.}

\item{burn.in}{number of samples to be discarded as a burn-in phase.}

\item{thinning}{factor for reducing the autocorrelation of random points.}
}
\value{
It returns a matrix of dimensions \eqn{n}x\eqn{p} with the random points sampled.
}
\description{
This function generates observations from a truncated multivariate elliptical distribution
with location parameter \code{mu}, scale matrix \code{Sigma}, lower and upper
truncation points \code{lower} and \code{upper} via Slice Sampling algorithm with Gibbs sampler steps.
}
\details{
The \code{dist} argument represents the truncated distribution to be used. The values are
\code{Normal}, \code{t}, \code{'t'}, \code{PE}, \code{PVII}, \code{Slash}, and \code{CN}, for the
truncated Normal, Student-t, Laplace, Power Exponential, Pearson VII, Slash, and Contaminated Normal distribution,
respectively.

The argument \code{nu} is a parameter or vector of parameters depending on the density generating
function (DGF). For the truncated Student-t, Power Exponential, and Slash distribution, \code{nu} is a
positive number. For the truncated Pearson VII, \code{nu} is a vector with the first element greater than
\eqn{p/2} and the second element a positive number. For the truncated Contaminated Normal distribution,
\code{nu} is a vector of length 2 assuming values between 0 and 1.

This function also allows generating random numbers from other truncated elliptical distributions not specified
in the \code{dist} argument, by supplying the density generating function (DGF) through arguments either
\code{expr} or \code{gFun}. The DGF must be a non-negative and strictly decreasing function on \code{(0, Inf)}.
The easiest way is to provide the DGF expression to argument \code{expr} as a character. The notation used in
\code{expr} needs to be understood by package \code{Ryacas0}, and the environment of \code{R}. For instance,
for the DGF \eqn{g(t)=e^{-t}}, the user must provide \code{expr = "exp(1)^(-t)"}. See that the function must depend
only on variable \eqn{t}, and any additional parameter must be passed as a fixed value. For this case, when a character
expression is provided to \code{expr}, the algorithm tries to compute a closed-form expression for the inverse function
of \eqn{g(t)}, however, this is not always possible (a warning message is returned). See example 2.

If it was no possible to generate random samples by passing a character expression to \code{expr}, the user may provide
a custom \code{R} function to the \code{gFun} argument. By default, its inverse function is approximated numerically,
however, the user may also provide its inverse to the \code{ginvFun} argument to gain some computational time.
When \code{gFun} is provided, arguments \code{dist} and \code{expr} are ignored.
}
\note{
The Normal distribution is a particular case of the Power Exponential distribution when \code{nu = 1}.
The Student-t distribution with \eqn{\nu} degrees of freedom results from the Pearson VII
distribution when \code{nu = } ((\eqn{\nu}+p)/2, \eqn{\nu}).
}
\examples{
library(ggplot2)
library(ggExtra)
library(gridExtra)

# Example 1: Sampling from the Truncated Normal distribution
set.seed(1234)
mu  = c(0, 1)
Sigma = matrix(c(1,0.70,0.70,3), 2, 2)
lower = c(-2, -3)
upper = c(3, 3)
sample1 = rtelliptical(5e4, mu, Sigma, lower, upper, dist="Normal")

# Histogram and density for variable 1
ggplot(data.frame(sample1), aes(x=X1)) +
   geom_histogram(aes(y=after_stat(density)), colour="black", fill="grey", bins=15) +
   geom_density(color="red") + labs(x=bquote(X[1]), y="Density") + theme_bw()

# Histogram and density for variable 2
ggplot(data.frame(sample1), aes(x=X2)) +
   geom_histogram(aes(y=after_stat(density)), colour="black", fill="grey", bins=15) +
   geom_density(color="red") + labs(x=bquote(X[2]), y="Density") + theme_bw()

\donttest{
# Example 2: Sampling from the Truncated Logistic distribution

# Function for plotting the sample autocorrelation using ggplot2
acf.plot = function(samples){
 p = ncol(samples); n = nrow(samples); q1 = qnorm(0.975)/sqrt(n); acf1 = list(p)
 for (i in 1:p){
   bacfdf = with(acf(samples[,i], plot=FALSE), data.frame(lag, acf))
   acf1[[i]] = ggplot(data=bacfdf, aes(x=lag,y=acf)) + geom_hline(aes(yintercept=0)) +
     geom_segment(aes(xend=lag, yend=0)) + labs(x="Lag", y="ACF", subtitle=bquote(X[.(i)])) +
     geom_hline(yintercept=c(q1,-q1), color="red", linetype="twodash") + theme_bw()
 }
 return (acf1)
}

set.seed(5678)
mu  = c(0, 0)
Sigma = matrix(c(1,0.70,0.70,1), 2, 2)
lower = c(-2, -2)
upper = c(3, 2)
# Sample autocorrelation with no thinning
sample2 = rtelliptical(10000, mu, Sigma, lower, upper, expr="exp(1)^(-t)/(1+exp(1)^(-t))^2")
grid.arrange(grobs=acf.plot(sample2), top="Logistic distribution with no thinning", nrow=1)

# Sample autocorrelation with thinning = 3
sample3 = rtelliptical(10000, mu, Sigma, lower, upper, expr="exp(1)^(-t)/(1+exp(1)^(-t))^2",
                       thinning=3)
grid.arrange(grobs=acf.plot(sample3), top="Logistic distribution with thinning = 3", nrow=1)
}

# Example 3: Sampling from the Truncated Kotz-type distribution
set.seed(5678)
mu  = c(0, 0)
Sigma = matrix(c(1,-0.5,-0.5,1), 2, 2)
lower = c(-2, -2)
upper = c(3, 2)
sample4 = rtelliptical(2000, mu, Sigma, lower, upper, gFun=function(t){t^(-1/2)*exp(-2*t^(1/4))})
f1 = ggplot(data.frame(sample4), aes(x=X1,y=X2)) + geom_point(size=0.50) +
     labs(x=expression(X[1]), y=expression(X[2]), subtitle="Kotz(2,1/4,1/2)") + theme_bw()
ggMarginal(f1, type="histogram", fill="grey")

}
\references{
{
  \insertRef{fang2018symmetric}{relliptical}

  \insertRef{ho2012some}{relliptical}

  \insertRef{neal2003slice}{relliptical}

  \insertRef{robert2010introducing}{relliptical}

  \insertRef{valeriano2023moments}{relliptical}
}
}
\seealso{
\code{\link{mvtelliptical}}
}
\author{
Katherine L. Valeriano, Christian E. Galarza and Larissa A. Matos
}
