% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/response.R
\docType{data}
\name{Response}
\alias{Response}
\alias{as.list.Response}
\alias{is.Response}
\title{HTTP Response handling}
\usage{
\method{as.list}{Response}(x, ...)

is.Response(x)
}
\arguments{
\item{x}{A \code{Response} object}

\item{...}{Ignored}
}
\value{
A rook-compliant list-response (in case of \code{as.list()}) or a logical
indicating whether the object is a \code{Response} (in case of \code{is.Response()})
}
\description{
This class handles all functionality involved in crafting a http response.
Much of the functionality is inspired by the Request class in Express.js, so
\href{https://expressjs.com/en/4x/api.html#res}{the documentation} for this will
complement this document. As \code{reqres} is build on top of the
\href{https://github.com/jeffreyhorner/Rook/blob/a5e45f751/README.md}{Rook specifications}
the \code{Response} object can be converted to a compliant list object to be
passed on to e.g. the \code{httpuv} handler.
}
\details{
A \code{Response} object is always created
as a response to a \code{Request} object and contains a reference to the
originating \code{Request} object. A \code{Response} is always initialized with a
404 Not Found code, an empty string as body and the \code{Content-Type} header set
to \code{text/plain}. As the \code{Content-Type} header is required for \code{httpuv} to
function, it will be inferred if missing when converting to a list. If the
body is a raw vector it will be set to \code{application/octet-stream} and
otherwise it will be set to \code{text/plain}. It is always advised to consciously
set the \code{Content-Type} header though. The only exception is when attaching a
standard file where the type is inferred from the file extension
automatically. Unless the body is a raw vector it will automatically be
converted to a character vector and collapsed to a single string with \code{"\n"}
separating the individual elements before the \code{Response} object is converted
to a list (that is, the body can exist as any type of object up until the
moment where the \code{Response} object is converted to a list). To facilitate
communication between different middleware the \code{Response} object contains
a data store where information can be stored during the lifetime of the
response.
}
\section{Initialization}{

A new 'Response'-object is initialized using the \code{new()} method on the
generator:

\strong{Usage}
\tabular{l}{
\code{res <- Response$new(request)}
}

But often it will be provided by the request using the \code{respond()} method,
which will provide the response, creating one if it doesn't exist

\strong{Usage}
\tabular{l}{
\code{res <- request$respond()}
}

\strong{Arguments}
\tabular{lll}{
\code{request} \tab  \tab The \code{Request} object that the \code{Response} is responding to\cr
}
}

\section{Fields}{

The following fields are accessible in a \code{Response} object:

\describe{
\item{\code{status}}{Gets or sets the status code of the response. Is initialised
with \code{404L}}
\item{\code{body}}{Set or get he body of the response. If it is a character
vector with a single element named \code{'file'} it will be interpreted as the
location of a file. It is better to use the \code{file} field for creating a
response referencing a file as it will automatically set the correct
headers.}
\item{\code{file}}{Set or get the location of a file that should be used as the
body of the response. If the body is not referencing a file (but contains
something else) it will return \code{NULL}. The \code{Content-Type} header will
automatically be inferred from the file extension, if known. If unknown it
will defaults to \code{application/octet-stream}. If the file has no extension it
will be \code{text/plain}. Existence of the file will be checked.}
\item{\code{type}}{Get or sets the \code{Content-Type} header of the response based on
a file extension or mime-type.}
\item{\code{request}}{Get the original \code{Request} object that the object is
responding to.}
}
}

\section{Methods}{

The following methods are available in a \code{Response} object:

\describe{
\item{\code{set_header(name, value)}}{Sets the header given by \code{name}. \code{value}
will be converted to character. A header will be added for each element in
\code{value}. Use \code{append_header()} for setting headers without overwriting
existing ones.}
\item{\code{get_header(name)}}{Returns the header(s) given by \code{name}}
\item{\code{remove_header(name)}}{Removes all headers given by \code{name}}
\item{\code{has_header(name)}}{Test for the existence of any header given by
\code{name}}
\item{\code{append_header(name, value)}}{Adds an additional header given by
\code{name} with the value given by \code{value}. If the header does not exist yet it
will be created.}
\item{\code{set_data(key, value)}}{Adds \code{value} to the internal data store and
stores it with \code{key}}
\item{\code{get_data(key)}}{Retrieves the data stored under \code{key} in the internal
data store.}
\item{\code{remove_data(key)}}{Removes the data stored under \code{key} in the
internal data store.}
\item{\code{has_data(key)}}{Queries whether the data store has an entry given by
\code{key}}
\item{\code{attach(file, filename=basename(file), type=NULL)}}{Sets the body to
the file given by \code{file} and marks the response as a download by setting the
\code{Content-Disposition} to \code{attachment; filename=<filename>}. Use the \code{type}
argument to overwrite the automatic type inference from the file extension.}
\item{\code{status_with_text(code)}}{Sets the status to \code{code} and sets the body
to the associated status code description (e.g. \code{Bad Gateway} for \code{502L})}
\item{\code{set_cookie(name, value, encode = TRUE, expires = NULL, http_only = NULL, max_age = NULL, path = NULL, secure = NULL, same_site = NULL)}}{Adds
the cookie given by \code{name} to the given \code{value}, optionally url encoding it,
along with any additional directives. See
\url{https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Set-Cookie} for a
description of the different directives. If the cookie already exists it
will be overwritten. The validity of the directives will automatically be
checked. \code{expires} expects a POSIXct object, \code{http_only} and \code{secure} expect
a logical, \code{max_age} expects an integer, \code{path} a string, and \code{same_site}
either \code{"Lax"} or \code{"Strict"}}
\item{\code{remove_cookie(name)}}{Removes the cookie named \code{name} from the
response.}
\item{\code{has_cookie(name)}}{Queries whether the response contains a cookie
named \code{name}}
\item{\code{set_links(...)}}{Sets the \code{Link} header based on the named arguments
passed to \code{...}. The names will be used for the \code{rel} directive.}
\item{\code{format(..., autofail = TRUE, compress = TRUE)}}{Based on the
formatters passed in through \code{...} content negotiation is performed with
request and the preferred formatter is chosen. The \code{Content-Type} header is
set automatically. If \code{compress = TRUE} the
\code{compress()} method will be called after formatting. If an error is
encountered and \code{autofail = TRUE} the response will be set to \code{500}. If a
formatter is not found and \code{autofail = TRUE} the response will be set to
\code{406}. If formatting is successful it will return \code{TRUE}, if not it will
return \code{FALSE}}
\item{\code{compress(priority = c('gzip', 'deflate', 'br', 'identity'))}}{Based
on the provided priority, an encoding is negotiated with the request and
applied. The \code{Content-Encoding} header is set to the chosen compression
algorithm.}
\item{\code{as_list()}}{Converts the object to a list for further processing by
a Rook compliant server such as \code{httpuv}. Will set \code{Content-Type} header if
missing and convert a non-raw body to a single character string.}
}
}

\examples{
fake_rook <- test <- fiery::fake_request(
  'http://example.com/test?id=34632&question=who+is+hadley',
  content = 'This is elaborate ruse',
  headers = list(
    Accept = 'application/json; text/*',
    Content_Type = 'text/plain'
  )
)

req <- Request$new(fake_rook)
res <- Response$new(req)
res

# Set the body to the associated status text
res$status_with_text(200L)
res$body

# Infer Content-Type from file extension
res$type <- 'json'
res$type

# Prepare a file for download
res$attach(system.file('DESCRIPTION', package = 'reqres'))
res$type
res$body
res$get_header('Content-Disposition')

}
\seealso{
\code{\link{Request}} for handling http requests
}
\keyword{datasets}
