% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/select_rate.R
\name{select_rate}
\alias{select_rate}
\title{Select rate results based on a range of criteria}
\usage{
select_rate(x, method = NULL, n = NULL)
}
\arguments{
\item{x}{list. An object of class \code{convert_rate} or \code{convert_rate_select}.}

\item{method}{string. Method by which to select or reorder rate results. For
most methods matching results are \emph{retained} in the output. See Details.}

\item{n}{numeric. Number, percentile, or range of results to retain or omit
depending on \code{method}. Default is \code{NULL}, in which case some methods will
instead reorder the results. See Details.}
}
\value{
The output of \code{select_rate} is a \code{list} object which retains the
\code{convert_rate} class, with an additional \code{convert_rate_select} class
applied.

It contains two additional elements: \verb{$original} contains the original,
unaltered \code{convert_rate} object, which will be retained unaltered through
multiple selection operations, that is even after processing through the
function multiple times. \verb{$select_calls} contains the calls for every
selection operation that has been applied to the \verb{$original} object, from
the first to the most recent. These additional elements ensure the output
contains the complete, reproducible history of the \code{convert_rate} object
having been processed.
}
\description{
The functions in \code{respR} are powerful, but outputs can be large
and difficult to explore, especially when there are hundreds to thousands
of results, for example the output of \code{auto_rate} on large datasets, or the
outputs of \code{calc_rate.int} from long intermittent-flow experiments.

The \code{select_rate} function helps explore, reorder, and filter
\code{convert_rate} results according to various criteria. For example,
extracting only positive or negative rates, only the highest or lowest
rates, only those from certain data regions, and numerous other methods
that allow advanced filtering of results so the final selection of rates is
well-defined towards the research question of interest. This also allows
for highly consistent reporting of results and rate selection criteria.

Multiple selection criteria can be applied by saving the output and
processing it through the function multiple times using different methods,
or alternatively via piping (\verb{|>} or \verb{\%>\%}). See Examples.

\emph{Note:} when choosing a \code{method}, keep in mind that to remain
mathematically consistent, \code{respR} outputs oxygen consumption (i.e.
respiration) rates as negative values. This is particularly important in
the difference between \code{highest/lowest} and \code{minimum/maximum} methods. See
Details.

When a rate result is omitted by the selection criteria, it is removed from
the \verb{$rate.output} element of the \code{convert_rate} object, and the associated
data in \verb{$summary} (i.e. that row) is removed. Some methods can also be
used with an \code{n = NULL} input to reorder the \verb{$rate} and \verb{$summary}
elements in various ways.
\subsection{Replicate and Rank columns}{

The summary table \verb{$rank} column is context-specific, and what it
represents depends on the type of experiment analysed or the function used
to determine the rates. If numeric values were converted, it is the order
in which they were entered. Similarly, if \code{calc_rate} was used, it is the
order of rates as entered using \code{from} and \code{to} (if multiple rates were
determined). For \code{auto_rate} it relates to the \code{method} input. For example
it indicates the kernel density ranking if the \code{linear} method was used,
the ascending or descending ordering by absolute rate value if \code{lowest} or
\code{highest} were used, or the numerical order if \code{minimum} or \code{maximum} were
used. For intermittent-flow experiments analysed via \code{calc_rate.int} and
\code{auto_rate.int} these will be ranked \emph{within} each replicate as indicated
in the \verb{$rep} column. The \verb{$rep} and \verb{$rank} columns can be used to keep
track of selection or reordering because the original values will be
retained unchanged through selection or reordering operations. The original
order can always be restored by using \code{method = "rep"} or \code{method = "rank"}
with \code{n = NULL}. In both these cases the \verb{$summary} table and
\verb{$rate.output} will be reordered by \verb{$rep} (if used) then \verb{$rank} to
restore the original ordering.

Note that if you are analysing intermittent-flow data and used
\code{auto_rate.int} but changed the \code{n} input to output more than one rate
result per replicate, the selection or reordering operations will not take
any account of this. You should carefully consider if or why you need to
output multiple rates per replicate in the first place. If you have, you
can perform selection on individual replicates by using \code{method = "rep"} to
select individual replicates then apply additional selection criteria.
}
}
\details{
These are the current methods by which rates in \code{convert_rate}
objects can be selected. Matching results are \emph{retained} in the output.
Some methods can also be used to reorder the results. Note that the methods
selecting by rate value operate on the \verb{$rate.output} element, that is the
final converted rate value.
\subsection{\code{positive}, \code{negative}}{

Selects all \code{positive} (>0) or \code{negative} (<0) rates. \code{n} is ignored.
Useful, for example, in respirometry on algae where both oxygen consumption
and production rates are recorded. Note, \code{respR} outputs oxygen consumption
(i.e. respiration) rates as \emph{negative} values, production rates as
\emph{positive}.
}

\subsection{\code{nonzero}, \code{zero}}{

Retains all \code{nonzero} rates (i.e. removes any zero rates), or retains
\emph{only} \code{zero} rates (i.e. removes all rates with any value). \code{n} is
ignored.
}

\subsection{\code{lowest}, \code{highest}}{

These methods can only be used when rates all have the same sign, that is
are all negative or all positive. These select the lowest and highest
\emph{\strong{absolute}} rate values. For example, if rates are all negative, \code{method = 'highest'} will retain the highest magnitude rates regardless of the
sign. \code{n} should be an integer indicating the number of lowest/highest
rates to retain. If \code{n = NULL} the results will instead be reordered by
lowest or highest rate without any removed. See \code{minimum} and \code{maximum}
options for extracting \emph{numerically} lowest and highest rates.
}

\subsection{\code{lowest_percentile}, \code{highest_percentile}}{

These methods can also only be used when rates all have the same sign.
These retain the \code{n}'th lowest or highest percentile of \emph{\strong{absolute}} rate
values. For example, if rates are all negative \code{method = 'highest_percentile'} will retain the highest magnitude \code{n}'th percentile
regardless of the sign. \code{n} should be a percentile value between 0 and 1.
For example, to extract the lowest 10th percentile of absolute rate values,
you would enter \verb{method = 'lowest_percentile', n = 0.1}.
}

\subsection{\code{minimum}, \code{maximum}}{

In contrast to \code{lowest} and \code{highest}, these are \emph{strictly numerical}
options which take full account of the sign of the rate, and can be used
where rates are a mix of positive and negative. For example, \code{method = 'minimum'} will retain the minimum numerical value rates, which would
actually be the highest oxygen uptake rates. \code{n} is an integer indicating
how many of the min/max rates to retain. If \code{n = NULL} the results will
instead be reordered by minimum or maximum rate without any removed.
}

\subsection{\code{minimum_percentile}, \code{maximum_percentile}}{

Like \code{min} and \code{max} these are \emph{strictly numerical} inputs which retain the
\code{n}'th minimum or maximum percentile of the rates and take full account of
the sign. Here \code{n} should be a percentile value between 0 and 1. For
example, if rates are all negative (i.e. typical uptake rates), to extract
the lowest 10th percentile of rates, you would enter \verb{method = 'maximum_percentile', n = 0.1}. This is because the \emph{lowest} negative rates
are numerically the \emph{maximum} rates (\code{highest/lowest} percentile methods
would be a better option in this case however).
}

\subsection{\code{rate}}{

Allows you to enter a value range of output rates to be retained. Matching
regressions in which the rate value falls within the \code{n} range (inclusive)
are retained. \code{n} should be a vector of two values. For example, to retain
only rates where the \code{rate} value is between 0.05 and 0.08: \verb{method = 'rate', n = c(0.05, 0.08)}. Note this operates on the \verb{$rate.output}
element, that is converted rate values.
}

\subsection{\code{rep}, \code{rank}}{

These refer to the respective columns of the \verb{$summary} table. For these,
\code{n} should be a numeric vector of integers of \code{rep} or \code{rank} values to
retain. To retain a range use regular R syntax, e.g. \code{n = 1:10}. If \code{n = NULL} no results will be removed, instead the results will be reordered
ascending by \code{rep} (if it contains values) then \code{rank}. Essentially this
restores the original ordering if other reordering operations have been
performed.

The values in these columns depend on the functions used to calculate
rates. If \code{calc_rate} was used, \code{rep} is \code{NA} and \code{rank} is the order of
rates as entered using \code{from} and \code{to} (if multiple rates were determined).
For \code{auto_rate}, \code{rep} is \code{NA} and \code{rank} relates to the \code{method} input.
For example it indicates the kernel density ranking if the \code{linear} method
was used, the ascending or descending ordering by absolute rate value if
\code{lowest} or \code{highest} were used, or by numerical order if \code{minimum} or
\code{maximum} were used. If \code{calc_rate.int} or \code{auto_rate.int} were used, \code{rep}
indicates the replicate number and the \code{rank} column represents rank
\emph{within} the relevant replicate, and will generally be filled with the
value \code{1}. Therefore you need to adapt your selection criteria
appropriately towards which of these columns is relevant.
}

\subsection{\code{rsq}, \code{row}, \code{time}, \code{density}}{

These methods refer to the respective columns of the \verb{$summary} data frame.
For these, \code{n} should be a vector of two values. Matching regressions in
which the respective parameter falls within the \code{n} range (inclusive) are
retained. To retain all rates with a R-Squared 0.90 or above: \verb{method = 'rsq', n = c(0.9, 1)}. The \code{row} and \code{time} ranges refer to the
\verb{$row}-\verb{$endrow} or \verb{$time}-\verb{$endtime} columns and the original raw data
(\verb{$dataframe} element of the \code{convert_rate} object), and can be used to
constrain results to rates from particular regions of the data (although
usually a better option is to \code{subset_data()} prior to analysis). Note
\code{time} is not the same as \code{duration} - see later section - and \code{row} refers
to rows of the raw data, \strong{not} rows of the summary table - see \code{manual}
method for this. For all of these methods, if \code{n = NULL} no results will be
removed, instead the results will be reordered by that respective column
(descending for \code{rsq} and \code{density}, ascending for \code{row}, and \code{time}).
}

\subsection{\code{time_omit}, \code{row_omit}}{

These methods refer to the original data, and are intended to \emph{exclude}
rates determined over particular data regions. This is useful in the case
of, for example, a data anomaly such as a spike or sensor dropout. For
these inputs, \code{n} are values (a single value, multiple values, or a range)
indicating data timepoints or rows of the original data to exclude. Only
rates (i.e. regressions) which \emph{do not} utilise those particular values are
retained in the output. For example, if an anomaly occurs precisely at
timepoint 3000, \code{time_omit = 3000} means only rates determined solely over
regions before or after this will be retained. If it occurs over a range
this can be entered as, \code{time_omit = c(3000,3200)}. If you want to exclude
a regular occurrence, for example the flushes in intermittent-flow
respirometry, or any other non-continuous values they can be entered as a
vector, e.g. \code{row_omit = c(1000, 2000, 3000)}. Note this last option can be
extremely computationally intensive when the vector or dataset is large, so
should only be used when a range cannot be entered as two values, which is
much faster. For both methods, input values must match exactly to values
present in the dataset.
}

\subsection{\code{oxygen}}{

This can be used to constrain rate results to regions of the data based on
oxygen values. \code{n} should be a vector of two values in the units of oxygen
in the raw data. Only rate regressions in which all datapoints occur within
this range (inclusive) are retained. Any which use even a single value
outside of this range are excluded. Note the summary table columns \code{oxy}
and \code{endoxy} refer to the first and last oxygen values in the rate
regression, which should broadly indicate which results will be removed or
retained, but this method examines \emph{every} oxygen value in the regression,
not just first and last.
}

\subsection{\code{oxygen_omit}}{

Similar to \code{time_omit} and \code{row_omit} above, this can be used to \emph{omit}
rate regressions which use particular oxygen values. For this \code{n} are
values (single or multiple) indicating oxygen values in the original raw
data to exclude. Every oxygen value used by each regression is checked, and
to be excluded an \code{n} value must match \emph{exactly} to one in the data.
Therefore, note that if a regression is fit across the data region where
that value would occur, it is not necessarily excluded unless that \emph{exact
value} occurs. You need to consider the precision of the data values
recorded. For example, if you wanted to exclude any rate using an oxygen
value of \code{7}, but your data are recorded to two decimals, a rate fit across
these data would \emph{not} be excluded: \code{c(7.03, 7.02, 7.01, 6.99, 6.98, ...)}.
To get around this you can use regular R syntax to input vectors at the
correct precision, such as seq, e.g. \code{seq(from = 7.05, to = 6.96, by = -0.01)}. This can be used to input ranges of oxygen values to exclude.
}

\subsection{\code{duration}}{

This method allows selection of rates which have a specific duration range.
Here, \code{n} should be a numeric vector of two values. Use this to set minimum
and maximum durations in the time units of the original data. For example,
\code{n = c(0,500)} will retain only rates determined over a maximum of 500 time
units. To retain rates over a minimum duration, set this using the minimum
value plus the maximum duration or simply infinity. For example, for rates
determined over a minimum of 500 time units \code{n = c(500,Inf)})
}

\subsection{\code{manual}}{

This method simply allows particular rows of the \verb{$summary} data frame to
be manually selected to be retained. For example, to keep only the top row
\verb{method = 'manual', n = 1}. To keep multiple rows use regular \code{R} selection
syntax: \code{n = 1:3}, \code{n = c(1,2,3)}, \code{n = c(5,8,10)}, etc. No value of \code{n}
should exceed the number of rows in the \verb{$summary} data frame. Note this is
not necessarily the same as selecting by the \code{rank} method, as the table
could already have undergone selection or reordering.
}

\subsection{\code{overlap}}{

This method removes rates which overlap, that is regressions which are
partly or completely fit over the same rows of the original data. This is
useful in particular with \code{auto_rate} results. The \code{auto_rate} \code{linear}
method may identify multiple linear regions, some of which may
substantially overlap, or even be completely contained within others. In
such cases summary operations such as taking an average of the rate values
may be questionable, as certain values will be weighted higher due to these
multiple, overlapping results. This method removes overlapping rates, using
\code{n} as a threshold to determine degree of permitted overlap. It is
recommended this method be used after all other selection criteria have
been applied, as it is quite aggressive about removing rates, and can be
\emph{very} computationally intensive when there are many results.

While it can be used with \code{auto_rate} results determined via the \code{rolling},
\code{lowest}, or \code{highest} methods, by their nature these methods produce \emph{all
possible} overlapping regressions, ordered in various ways, so other
selection methods are more appropriate. The \code{overlap} method is generally
intended to be used in combination with the \code{auto_rate} \code{linear} results,
but may prove useful in other analyses.

Permitted overlap is determined by \code{n}, which indicates the proportion of
each particular regression which must overlap with another for it to be
regarded as overlapping. For example, \code{n = 0.2} means a regression would
have to overlap with at least one other by at least 20\% of its total length
to be regarded as overlapping.

The \code{overlap} method performs two operations:

First, regardless of the \code{n} value, any rate regressions which are
completely contained within another are removed (this is also the only
operation if \code{n = 1}).

Secondly, for each regression in \verb{$summary} starting from the bottom of the
summary table (usually the lowest ranked result, but this depends on the
analysis used and if any reordering has been already occurred), the
function checks if it overlaps with any others (accounting for \code{n}). If
not, the next lowest is checked, and the function progresses up the summary
table until it finds one that does. The first to be found overlapping is
then removed, and the process repeats starting again from the bottom of the
summary table. If no reordering to the results has occurred, this means
lower ranked results are removed first. This is repeated iteratively until
only non-overlapping rates (accounting for \code{n}) remain.

If \code{n = 0}, only rates which do not overlap at all, that is share \emph{no}
data, are retained. If \code{n = 1}, only rates which are 100\% contained within
at least one other are removed.
}

\subsection{Reordering results}{

Several methods can be used to reorder results rather than select them, by
not entering an \code{n} input (that is, letting the \code{n = NULL} default be
applied). Several of these methods are named the same as those in
\code{auto_rate} for consistency and have equivalent outcomes, so this allows
results to be reordered to the equivalent of that method's results without
re-running the \code{auto_rate} analysis.

The \code{"row"} and \code{"rolling"} methods reorder sequentially by the starting
row of each regression (\verb{$row} column).

The \code{"time"} method reorders sequentially by the starting time of each
regression (\verb{$time} column).

\code{"linear"} and \code{"density"} are essentially identical, reordering by the
\verb{$density} column. This metric is only produced by the \code{auto_rate} \code{linear}
method, so will not work with any other results.

\code{"rep"} or \code{"rank"} both reorder by the \verb{$rep} then \verb{$rank} columns. What
these represents is context dependent - see \strong{Replicate and Rank columns}
section above. Each summary row \code{rep} and \code{rank} value is retained
unchanged regardless of how the results are subsequently selected or
reordered, so this will restore the original ordering after other methods
have been applied.

\code{rsq} reorders by \verb{$rsq} from highest value to lowest.

\code{highest} and \code{lowest} reorder by absolute values of the \verb{$rate.output}
column, that is highest or lowest in magnitude regardless of the sign. They
can only be used when rates all have the same sign.

\code{maximum} and \code{minimum} reorder by numerical values of the \verb{$rate.output}
column, that is maximum or minimum in numerical value taking account of the
sign, and can be used when rates are a mix of negative and positive.
}

\subsection{Numeric input conversions}{

For \code{convert_rate} objects which contain rates which have been converted
from numeric values, the summary table will contain a limited amount of
information, so many of the selection or reordering methods will not work.
In this case a warning is given and the original input is returned.
}

\subsection{Plot}{

There is no plotting functionality in \code{select_rate}. However since the
output is a \code{convert_rate} object it can be plotted. See the \strong{Plot}
section in \code{help("convert_rate")}. To plot straight after a selection
operation, pipe or enter the output in \code{plot()}. See Examples.
}

\subsection{More}{

This help file can be found online
\href{https://januarharianto.github.io/respR/reference/select_rate.html}{here},
where it is much easier to read.

For additional help, documentation, vignettes, and more visit the \code{respR}
website at \url{https://januarharianto.github.io/respR/}
}
}
\examples{
\donttest{
## Object to filter
 ar_obj <- inspect(intermittent.rd, plot = FALSE) |>
   auto_rate(plot = FALSE) |>
   convert_rate(oxy.unit = "mg/L",
                time.unit = "s",
                output.unit = "mg/h",
                volume = 2.379) |>
   summary()

 ## Select only negative rates
 ar_subs_neg <- select_rate(ar_obj, method = "negative") |>
   summary()

 ## Select only rates over 1000 seconds duration
 ar_subs_dur <- select_rate(ar_obj, method = "duration", n = c(1000, Inf)) |>
   summary()

 ## Reorder rates sequentially (i.e. by starting row)
 ar_subs_dur <- select_rate(ar_obj, method = "row") |>
   summary()

 ## Select rates with r-squared higher than 0.99,
 ## then select the lowest 10th percentile of the remaining rates,
 ## then take the mean of those
 inspect(squid.rd, plot = FALSE) |>
   auto_rate(method = "linear",
             plot = FALSE) |>
   convert_rate(oxy.unit = "mg/L",
                time.unit = "s",
                output.unit = "mg/h",
                volume = 2.379) |>
   summary() |>
   select_rate(method = "rsq", n = c(0.99, 1)) |>
   select_rate(method = "lowest_percentile", n = 0.1) |>
   mean()
   }
}
