% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ee_download.R
\name{ee_gcs_to_local}
\alias{ee_gcs_to_local}
\title{Move results from Google Cloud Storage to a local directory}
\usage{
ee_gcs_to_local(
  task,
  dsn,
  public = FALSE,
  metadata = FALSE,
  overwrite = TRUE,
  quiet = FALSE
)
}
\arguments{
\item{task}{List generated after finished an EE task correctly. See details.}

\item{dsn}{Character. Output filename. If missing, a temporary
file (i.e. \code{tempfile()}) is assigned.}

\item{public}{Logical. If TRUE, a public link to Google Cloud Storage
resource is created.}

\item{metadata}{Logical. If TRUE, export the metadata related to the Google
Cloud Storage resource. See details.}

\item{overwrite}{A boolean argument that indicates indicating
whether "filename" should be overwritten. By default TRUE.}

\item{quiet}{Logical. Suppress info message}
}
\value{
If \code{metadata} is FALSE, will return the filename of the Google
Cloud Storage resource on their system. Otherwise, a list with two elements
(\code{dns} and \code{metadata}) is returned.
}
\description{
Move results of an EE task saved in Google Cloud Storage to a local
directory.
}
\details{
The task argument needs "COMPLETED" task state to work due to that the parameters
necessaries to locate the file into Google Cloud Storage are obtained from \cr
\code{ee$batch$Export$*$toCloudStorage(...)$start()$status()}.

If the argument \code{metadata} is TRUE, a list with the
following elements is exported join with the output filename (dsn):

\itemize{
\item{\bold{ee_id: }}{Name of the Earth Engine task.}
\item{\bold{gcs_name: }}{Name of the Table in Google Cloud Storage.}
\item{\bold{gcs_bucket: }}{Name of the bucket.}
\item{\bold{gcs_fileFormat: }}{Format of the table.}
\item{\bold{gcs_public_link: }}{Download link to the table.}
\item{\bold{gcs_URI: }}{gs:// link to the table.}
}
}
\examples{
\dontrun{
library(rgee)
library(stars)
library(sf)

ee_users()
ee_Initialize(gcs = TRUE)

# Define study area (local -> earth engine)
# Communal Reserve Amarakaeri - Peru
rlist <- list(xmin = -71.13, xmax = -70.95,ymin = -12.89, ymax = -12.73)
ROI <- c(rlist$xmin, rlist$ymin,
         rlist$xmax, rlist$ymin,
         rlist$xmax, rlist$ymax,
         rlist$xmin, rlist$ymax,
         rlist$xmin, rlist$ymin)
ee_ROI <- matrix(ROI, ncol = 2, byrow = TRUE) \%>\%
  list() \%>\%
  st_polygon() \%>\%
  st_sfc() \%>\%
  st_set_crs(4326) \%>\%
  sf_as_ee()


# Get the mean annual NDVI for 2011
cloudMaskL457 <- function(image) {
  qa <- image$select("pixel_qa")
  cloud <- qa$bitwiseAnd(32L)$
    And(qa$bitwiseAnd(128L))$
    Or(qa$bitwiseAnd(8L))
  mask2 <- image$mask()$reduce(ee$Reducer$min())
  image <- image$updateMask(cloud$Not())$updateMask(mask2)
  image$normalizedDifference(list("B4", "B3"))
}

ic_l5 <- ee$ImageCollection("LANDSAT/LT05/C01/T1_SR")$
  filterBounds(ee$FeatureCollection(ee_ROI))$
  filterDate("2011-01-01", "2011-12-31")$
  map(cloudMaskL457)

# Create simple composite
mean_l5 <- ic_l5$mean()$rename("NDVI")
mean_l5 <- mean_l5$reproject(crs = "EPSG:4326", scale = 500)
mean_l5_Amarakaeri <- mean_l5$clip(ee_ROI)

# Move results from Earth Engine to Drive
task_img <- ee_image_to_gcs(
   image = mean_l5_Amarakaeri,
   bucket = "rgee_dev",
   fileFormat = "GEO_TIFF",
   region = ee_ROI,
   fileNamePrefix = "my_image_demo"
)

task_img$start()
ee_monitoring(task_img)

# Move results from Drive to local
img <- ee_gcs_to_local(task = task_img)
}
}
\seealso{
Other generic download functions: 
\code{\link{ee_drive_to_local}()}
}
\concept{generic download functions}
