\name{map.z}
\alias{map.z}
\title{ Plot a Map of Data using Proportional Symbols }
\description{
Displays a simple map where the data are represented by open circles whose diameters are proportional to the value of the data at their spatial locations.  The rate of change of symbol diameter with value and the absolute size of the symbols are defined by the user.  Optionally a legend may be displayed on the map.
}
\usage{
map.z(xx, yy, zz, p = 0.5, sfact = 2.5, zmin = NA, zmax = NA, 
	xlab = "Easting", ylab = "Northing",
	zlab = deparse(substitute(zz)), main = "", tol = 0.04,
	symcolr = 1, ifparams = FALSE, iflgnd = FALSE,
	title = deparse(substitute(zz)), ...)
}
\arguments{
  \item{xx}{ name of the x-axis spatial coordinate, the eastings. }
  \item{yy}{ name of the y-axis spatial coordinate, the northings. }
  \item{zz}{ name of the variable to be plotted as a map. }
  \item{p}{ a parameter that controls the rate of change of symbol diameter with changing value.  A default of \code{p = 0.5} is provided.  See Details below. }
  \item{sfact}{ controls the absolute size of the plotted symbols, by default \code{sfact = 2.5}.  Increasing sfact results in larger symbols. }
  \item{zmin}{ a value below which all symbols will be plotted at the same minumum size.  By default \code{zmin = NA} which results in the minimum value of the variable defining the minimum symbol size.  See Details below. }
  \item{zmax}{ a value above which all symbols will be plotted at the same maximum size.  By default \code{zmax = NA} which results in the maximum value of the variable defining the maximum symbol size.  See Details below. }
  \item{xlab}{ a title for the x-axis, defaults to \code{Easting}. }
  \item{ylab}{ a title for the y-axis, defaults to \code{Northing}. }
  \item{zlab}{ by default, \code{zlab = deparse(substitute(zz))}, a map title is generated by appending the input variable name text string to \code{"Proportional Symbol Map for "}. Alternative titles may be provided, see Details below. }
  \item{main}{ an alternative map title, see Details below.}
  \item{tol}{ a parameter used to ensure the area included within the neatline around the map is larger than the distribution of the points so that the plotted symbols fall within the neatline.  By default \code{tol = 0.04}, if more clearance is required increase the value of \code{tol}. }
  \item{symcolr}{ the colour of the symbols, the default is black, \code{symcolr = 1}.  This may be changed if required, see \code{\link{display.lty}} for the default colour palette.  For example, \code{symcolr = 2} will cause the symbols to be plotted in red. }
  \item{ifparams}{if \code{ifparams = TRUE} on completion of plotting and after the legend has been plotted, if requested, the cursor is activated, locate that at the top left corner of the desired text position and \sQuote{left button} on the pointing device.  This text comprises three lines: the values of \code{p} to three significant figures and \code{sfact}; the maximum value of \code{z} to 3 significant figures and \code{zmax}; and the minimum value of \code{z} to 3 significant figures and \code{zmin}.  The default is no text display. }
  \item{iflgnd}{ the default is no legend.  If a legend is required set \code{iflgnd = TRUE}, following the plotting of the data the cursor will be activated, locate that at the top left corner of the desired legend position and \sQuote{left button} on the pointing device.  See Notes below.}
  \item{title}{ a short title for the legend, e.g., \code{title = "Zn (mg/kg)"}.  The default is the variable name. } 
  \item{\dots}{ further arguments to be passed to methods.  For example, if smaller plotting characters are required for the legend, specify, for example, \code{cex = 0.8};  and if some other colour than black is required for the legend, specify, for example, \code{col = 3}, to obtain blue.  Any colour change will also be reflected in the legend, if displayed.  See \code{\link{display.lty}} for the default colour palette.  If it is required to make the map title smaller, add \code{cex.main = 0.9} to reduce the font size by 10\%. }
}
\details{
The symbol diameter is computed as a function of the value \code{z} to be plotted:\cr
\code{diameter = dmin + (dmax - dmin) * {(z - zmin)/(zmax - zmin)}^p}\cr
where \code{dmin} and \code{dmax} are defined as 0.1 and 1 units, so the symbol diameters range over an order of magnitude (and symbol areas over two); \code{zmin} and \code{zmax} are the observed range of the data, or the range over which the user wants the diameters to be computed; and \code{p} is a power defined by the user.  The value of \code{(z - zmin)/(zmax - zmin)} is the value of \code{z} normalized, 0 - 1, to the range over which the symbol diameters are to be computed.  After being raised to the power \code{p}, which will result in a number in the range 0 to 1, this value is multiplied by the permissable range of diameters and added to the minimum diameter.  This results in a diameter between 0.1 and 1 units that is proportional to the value of \code{z}.

A \code{p} value of \code{1} results in a linear rate of change.  Values of \code{p} less than unity lead to a rapid intial rate of change with increasing value of \code{z} which is often suitable for displaying positively skewed data sets, see the example below.  In contrast, values of \code{p} greater than unity result in an initial slow rate of change with increasing value of \code{z} which is often suitable for displaying negatively skewed data sets.  Experimentation is usually necessary to obtain a satisfactory visual effect.  See \code{\link{syms.pfunc}} for a graphic demonstrating the effect of varying the \code{p} parameter.

The user may choose to transform the variable to be plotted prior to determining symbol size etc., e.g., \code{log10(zz)}, to generate a logarithmic rate of symbol size change. See Example below.

If \code{zmin} or \code{zmax} are defined this has the effect of setting a minimum or maximum value of \code{z}, respectively, beyond which changes in the value of \code{z} do not result in changes in symbol diameter.  This can be useful in limiting the effect of one, or a few, extreme outlier(s) while still plotting them, they simply plot at the minimum or maximum symbol size and are not involved in the calculation of the range of \code{z} over which the symbol diameters vary.  \bold{Note:} If the variable \code{z} includes a transform, e.g., \code{log10(z)}, the values of \code{zmin} and/or \code{zmax} must be in those transform units. 

If \code{zlab} and \code{main} are undefined a default a map title is generated by appending the input variable name text string to \code{"Proportional Symbol Map for "}.  If no map title is required set \code{zlab = ""}, and if some user defined map title is required it should be defined in \code{main}, e.g. \code{main = "Map Title Text"}. 
}
\note{
Any less than detection limit values represented by negative values, or zeros or other numeric codes representing blanks in the data, must be removed prior to executing this function, see \code{\link{ltdl.fix.df}}.

The legend consists of five proportional symbols and their corresponding \code{z} values: \code{zmin}; the three quartiles; and \code{zmax}.  If \code{zmin} and \code{zmax} have been user defined it is over their range that the symbol sizes are computed and displayed.  When defining \code{zmin} and/or \code{zmax} it is useful to set \code{ifparams = TRUE} as a reminder, whilst developing the required display.

Any \code{NA}s in the data vector are removed prior to displaying the plot.

In some R installations the generation of multi-panel displays and the use of function eqscplot from package MASS causes warning messages related to graphics parameters to be displayed on the current device. These may be suppressed by entering \code{options(warn = -1)} on the R command line, or that line may be included in a \sQuote{first} function prepared by the user that loads the \sQuote{rgr} package, etc.
}
\author{ Robert G. Garrett }
\seealso{ \code{\link{syms}}, \code{\link{syms.pfunc}}, \code{\link{ltdl.fix.df}}, \code{\link{remove.na}} }  
\examples{ 
## Make test data available
data(kola.o)
attach(kola.o)

## Plot a default symbol map, p = 0.5 and sfact = 2.5
map.z(UTME, UTMN, Cu)

## Plot a map where the symbols are logarithmically scaled,
## and more appropriately labelled axes
map.z(UTME/1000, UTMN/1000, log10(Cu), p = 1,
	xlab = "Kola Project UTM Eastings (km)",
	ylab = "Kola Project UTM Northings (km)" )


## Plot with differently scaled symbols and more appropriately 
## labelled axes
map.z(UTME/1000, UTMN/1000, Cu, p = 0.3, sfact = 2.0,
	xlab = "Kola Project UTM Eastings (km)",
	ylab = "Kola Project UTM Northings (km)" ) 

## Plot a map as above but where outliers above a value of 1000 are
## displayed with the same symbol
map.z(UTME/1000, UTMN/1000, Cu, p = 0.3, sfact = 2.0, zmax = 1000,
	xlab = "Kola Project UTM Eastings (km)",
	ylab = "Kola Project UTM Northings (km)" )

## Detach test data
detach(kola.o) 
}
\keyword{ hplot }


