% Generated by roxygen2 (4.0.2): do not edit by hand
\name{ma.filter}
\alias{ma.filter}
\title{Refined Moving Average Filter}
\usage{
ma.filter(x, q = NULL, seasonal = FALSE, period = NULL, plot = TRUE)
}
\arguments{
\item{x}{a numeric vector or univariate time series.}

\item{seasonal}{a logical value indicating to estimate the seasonal component. Only valid
for \code{seasonal = TRUE}. The default is \code{FALSE}.}

\item{q}{specified moving average lag \eqn{q}. The default is \code{NULL}.}

\item{period}{seasonal period. The default is \code{NULL}.}

\item{plot}{a logical value indicating to make the plots. The default is \code{TRUE}.}
}
\value{
A matrix containing the following columns:
\item{data}{original data \code{x}.}
\item{trend}{fitted trend.}
\item{season}{seasonal indices if \code{seasonal = TRUE}.}
\item{residual}{irregularity or residuals.}
}
\description{
uses refined moving average filter to estimate the
trend component, and then obtain seasonal component if necessary.
}
\details{
For univariate time seties \eqn{x[t]}, the additive seasonal model is assumed
to be \deqn{x[t] = m[t] + S[t] + R[t],} where \eqn{m[t], S[t], R[t]} are trend, seasonal
and irregular components, respectively. Once we
obtain the optimal moving average lag \eqn{q} using \code{\link{qn}}, the trend can be
estimated by using the refined moving average
\deqn{mhat[t] = \sum x[t]/(2q+1),} for \eqn{q + 1 \le t \le n - q}.
If \eqn{q + 1 > n - q}, we take \eqn{q = min(n - q, q)}.
If there is no seasonal component, the irregularity or residuals can be computed by
\eqn{Rhat[t] = x[t] - mhat[t]}. Otherwise, the seasonal index \eqn{Shat[t]} can be
estimated by averaging the sequence \eqn{x[t] - mhat[t]} for each of \code{1:period}. For
example, the seasonal component in January can be estimated by the average of all of the
observations made in January after removing the trend component.
To ensure the identifiability of
\eqn{m[t]} and \eqn{S[t]}, we have to assume
\deqn{S[i + j*period] = S[i],  \sum S[i] = 0,}
where \eqn{i = 1,..., period; j = floor(n/period)}.
The irregularity or residuals are then computed by
\eqn{Rhat[t] = x[t] - mhat[t] - Shat[t]}. For \eqn{t < q + 1} and
\eqn{t > n - q}, the corresponding estimators are based on equation (7) in D. Qiu
\emph{et al.} (2013). More details about estimating the trend component can be seen
in Section 1.5 of P.J. Brockwell \emph{et al.} (1991) or Chapter 6 of J. Fan \emph{et al.} (2003).

For the multiplicative seasonal model \deqn{x[t] = m[t] * S[t] * R[t],} it can be
transformed to an additive seasonal model by taking a logarithm on both sides if
\eqn{x[t] > 0}, i.e., \deqn{log(x[t]) = log(m[t]) + log(S[t]) + log(R[t]),} and then use
the refined moving average filter for the components decomposition as the same
in the additive seasonal model.

Plots of original data v.s fitted data, fitted trend, seasonal indices (if \code{seasonal = TRUE}) and residuals will be drawn if \code{plot = TRUE}.
}
\examples{
## decompose the trend for the first difference of annual global air temperature from 1880-1985
data(globtemp)
decomp1 <- ma.filter(globtemp)

## decompose the trend and seasonality for CO2 data with monthly and additive seasonality
decomp2 <- ma.filter(co2, seasonal = TRUE, period = 12)

## decompose the trend and seasonality for monthly airline passenger numbers from 1949-1960
decomp3 <- ma.filter(AirPassengers, seasonal = TRUE, period = 12)

## simulation data: oracally efficient estimation for AR(p) coefficients
d <- 12
n <- d*100
x <- (1:n)/n
y <- 1 + 2*x + 0.3*x^2 + sin(pi*x/6) + arima.sim(n = n,list(ar = 0.2), sd = 1)
fit <- ma.filter(y,seasonal = TRUE,period = 12,plot = FALSE)
ar(fit[,4], aic = FALSE, order.max = 1)$ar
}
\author{
Debin Qiu
}
\references{
D. Qiu, Q. Shao, and L. Yang (2013), Efficient inference for autoregressive coeficient
in the presence of trend. \emph{Journal of Multivariate Analysis} 114, 40-53.

J. Fan and Q. Yao, Nonlinear Time Series: Nonparametric and Parametric Methods, first ed.,
Springer, New York, 2003.

P.J. Brockwell, R.A. Davis, Time Series: Theory and Methods, second ed., Springer,
New York, 1991.
}
\seealso{
\code{\link{ss.filter}}
}

