# --------------------------------------
# Author: Andreas Alfons
#         Erasmus Universiteit Rotterdam
# --------------------------------------

#' Robustness weights of Huber M-estimation of location and scatter
#' 
#' Extract (relative) robustness weights of a Huber M-estimate of location and 
#' scatter.
#' 
#' @method weights covHuber
#' 
#' @param object  an object inheriting from class \code{"\link{covHuber}"} 
#' containing Huber M-estimates of location and scatter.
#' @param type  a character string specifying the type of robustness weights to 
#' be extracted.  Possible values are \code{"consistent"} and 
#' \code{"relative"}.  The former can be used for a robust transformation of 
#' the data such that the covariance matrix of the transformed data is Fisher 
#' consistent.  Observations that are not downweighted in general receive a 
#' weight larger than 1.  The latter are useful for interpretation, as 
#' observations that are not downweighted receive a relative weight of 1.
#' @param \dots  additional arguments are currently ignored.
#' 
#' @return A numeric vetor containing the requested robustness weights.
#' 
#' @author Andreas Alfons
#' 
#' @references
#' Zu, J. and Yuan, K.-H. (2010) Local influence and robust procedures for 
#' mediation analysis. \emph{Multivariate Behavioral Research}, \bold{45}(1), 
#' 1--44.
#' 
#' @seealso \code{\link{covHuber}}
#' 
#' @keywords utilities
#' 
#' @export

weights.covHuber <- function(object, type = c("consistent", "relative"), ...) {
  # initializations
  type <- match.arg(type)
  # extract weights
  weights <- object$weights
  if(type == "consistent") weights <- weights / sqrt(object$tau)
  weights
}
