#
# Script to test roxut package action on package roxutTestPkg
#
# This script roxygenizes, builds and checks roxutTestPkg as a means
# of ensuring that roxut functions as intended.
#
# This script makes a copy of roxutTestPkg in a temporary directory
# so that the original roxutTestPkg is not modified in situ.
#

out <- tryCatch(
  {
    message("\n\n>>> Processing roxutTestPkg\n")
    message("\n>>> Setting up temporary directory for testing\n")

    # Step 0.  Set up testing environment in a temp dir
    cur_dir <- getwd()
    setwd("roxutTestPkg")
    td <- tempdir()
    tst_dir <- paste(td, "roxutTestPkg", sep = "/")
    dir.create(tst_dir)
    system2("cp", paste0("-r . ", tst_dir))
    setwd(tst_dir)

    message(paste("tempdir path:", td, sep = " "))

    # Step 1. Run roxygenize with the tests_roclet

    message("\n>>> roxygenizing using rd, namespace, collate & tests roclets\n")

    roxygenize(roclets = c("rd", "namespace", "collate", "tests_roclet"))

    # Step 2. Verify that roxut updated the unit test files in roxutTestPkg
    # When roxygenize() was run above the package roxutTestPkg was loaded.

    tinytest_files <- list.files("inst/tinytest", "test_.*\\.R", full.names = TRUE)
    for (i in 1:length(tinytest_files)) {
      tmp <- readLines(tinytest_files[i])[1]
      status <- grepl("roxut", tmp)
      if (!status) warning(paste("Warning: file", tinytest_files[i], "was not generated by roxut", sep = " "))
      if (status) message(paste("Confirming file", tinytest_files[i], "was generated by roxut", sep = " "))
    }

    testthat_files <- list.files("tests/testthat", "test_.*\\.R", full.names = TRUE)
    for (i in 1:length(testthat_files)) {
      tmp <- readLines(testthat_files[i])[1]
      status <- grepl("roxut", tmp)
      if (!status) warning(paste("Warning: file", testthat_files[i], "was not generated by roxut", sep = " "))
      if (status) message(paste("Confirming file", testthat_files[i], "was generated by roxut", sep = " "))
    }

    # system2("cp", "-r . ~/Desktop/tempdir") # good for troubleshooting

    # Step 3. Build and check package
    # See WRE sec 1.6 for the need to call R this (safe) way
    rhome <- Sys.getenv("R_HOME")
    localR <- paste(rhome, "bin", "R", sep = "/")
    setwd("..") # move up for the next actions

    message("\n>>> Building & checking roxutTestPkg\n")

    system2(localR, "CMD build roxutTestPkg")
    system2(localR, "CMD check roxutTestPkg_0.1.tar.gz")

    # Step 4. Check the output file for errors

    message("\n>>> Final inspection for problems\n")

    problem <- FALSE
    setwd("roxutTestPkg.Rcheck/tests")
    lines <- readLines("tinytest.Rout") # reading from an open file!
    if (any(grepl("There.*roxutTestPkg", lines))) problem <- TRUE
    if (!problem) message("tinytest.Rout is clean/no errors")
    if (problem) {
      message("tinytest.Rout has an error")
      stop("Encountered a problem while running test_roxutTestPkg.R")
    }
  },

  error = function(cond) {
    errmess <- "There was a problem processing & testing roxutTestPkg"
    message("\nError message from R: ", cond$message, "\n")
    message(errmess)
    return(NA)
  }
) # end of tryCatch
