# Copyright (C) 2011  Egon Willighagen <egon.willighagen@gmail.com>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as
# published by the Free Software Foundation, either version 3 of the
# License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details. 
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

new.rdf <- function(ontology=TRUE) {
    if (ontology) {
		model <- .jcall(
			"com/github/egonw/rrdf/RJenaHelper",
			"Lcom/hp/hpl/jena/rdf/model/Model;",
			"newOntoRdf"
		)
	} else {
		model <- .jcall(
			"com/github/egonw/rrdf/RJenaHelper",
			"Lcom/hp/hpl/jena/rdf/model/Model;",
			"newRdf"
		)
	}
	return(model)
}

load.rdf <- function(filename, format="RDF/XML", appendTo=NULL) {
	formats = c("RDF/XML", "TURTLE", "N-TRIPLES", "N3")
	if (!(format %in% formats))
		stop("Formats must be one in: ", formats)
	if (is.null(appendTo)) {
	    model <- .jcall(
    	    "com/github/egonw/rrdf/RJenaHelper",
       		"Lcom/hp/hpl/jena/rdf/model/Model;",
       		"loadRdf", filename, format
    	)
        return(model)
    } else {
	    model <- .jcall(
    	    "com/github/egonw/rrdf/RJenaHelper",
       		"Lcom/hp/hpl/jena/rdf/model/Model;",
       		"loadRdf", filename, format, appendTo
    	)
    }
}

save.rdf <- function(store, filename, format="RDF/XML") {
	formats = c("RDF/XML", "RDF/XML-ABBREV", "N3")
	if (!(format %in% formats))
		stop("Formats must be one in: ", formats)
	if (format == "RDF/XML") format <- "RDF/XML-ABBREV";
	.jcall(
			"com/github/egonw/rrdf/RJenaHelper",
			"V",
			"saveRdf", store, filename, format
	)
}

combine.rdf <- function(model1, model2) {
    if (attr(model2, "jclass") != "Lcom/hp/hpl/jena/rdf/model/Model") {
        model2 <- .jcast(model2, "com/hp/hpl/jena/rdf/model/Model")
    }

    .jcall(
        model1, "Lcom/hp/hpl/jena/rdf/model/Model;", 
        "add", model2
    )
}

summarize.rdf <- function(model) {
    count <- .jcall(
        "com/github/egonw/rrdf/RJenaHelper",
        "I", "tripleCount", model
    )
    exception <- .jgetEx(clear = TRUE)
    if (!is.null(exception)) {
        stop(exception)
    }
    print(paste("Number of triples:", count))
}

sparql.rdf <- function(model, sparql) {
    stringMat <- .jcall(
        "com/github/egonw/rrdf/RJenaHelper",
        "Lcom/github/egonw/rrdf/StringMatrix;", "sparql", model, sparql
    )
    exception <- .jgetEx(clear = TRUE)
    if (!is.null(exception)) {
        stop(exception)
    }
    return(.stringMatrix.to.matrix(stringMat))
}

sparql.remote <- function(endpoint, sparql) {
	stringMat <- .jcall(
			"com/github/egonw/rrdf/RJenaHelper",
			"Lcom/github/egonw/rrdf/StringMatrix;", "sparqlRemote", endpoint, sparql
	)
	exception <- .jgetEx(clear = TRUE)
	if (!is.null(exception)) {
		stop(exception)
	}
	return(.stringMatrix.to.matrix(stringMat))
}

add.triple <- function(store,
	subject="http://example.org/Subject",
	predicate="http://example.org/Predicate",
	object="http://example.org/Object") {
	.jcall(
		"com/github/egonw/rrdf/RJenaHelper",
		"V",
		"addObjectProperty", store,
		subject, predicate, object
	)
}

add.data.triple <- function(store,
		subject="http://example.org/Subject",
		predicate="http://example.org/Predicate",
		data="Value",
		type=NULL) {
	if (is.null(type)) {
		.jcall(
			"com/github/egonw/rrdf/RJenaHelper",
			"V",
			"addDataProperty", store,
			subject, predicate, data
		)
	} else {
		.jcall(
			"com/github/egonw/rrdf/RJenaHelper",
			"V",
			"addDataProperty", store,
			subject, predicate, data,
			type
		)
	}
}

construct.rdf <- function(model, sparql) {
	newModel <- .jcall(
		"com/github/egonw/rrdf/RJenaHelper",
		"Lcom/hp/hpl/jena/rdf/model/Model;",
		"construct", model, sparql
	)
	exception <- .jgetEx(clear = TRUE)
	if (!is.null(exception)) {
		stop(exception)
	}
	return(newModel)
}

construct.remote <- function(endpoint, sparql) {
	newModel <- .jcall(
			"com/github/egonw/rrdf/RJenaHelper",
			"Lcom/hp/hpl/jena/rdf/model/Model;",
			"constructRemote", endpoint, sparql
	)
	exception <- .jgetEx(clear = TRUE)
	if (!is.null(exception)) {
		stop(exception)
	}
	return(newModel)
}

add.prefix <- function(store=NULL, prefix=NULL, namespace=NULL) {
    if (is.null(store)) stop("A store must be given.")
    if (is.null(prefix)) stop("A prefix must be given.")
    if (is.null(namespace)) stop("A namespace must be given.")

	.jcall(
		"com/github/egonw/rrdf/RJenaHelper",
		"V",
		"addPrefix", store, prefix, namespace
	)
	exception <- .jgetEx(clear = TRUE)
	if (!is.null(exception)) {
		stop(exception)
	}
}

.stringMatrix.to.matrix <- function(stringMatrix) {
    nrows <- .jcall(stringMatrix, "I", "getRowCount")
    ncols <- .jcall(stringMatrix, "I", "getColumnCount")
    if (ncols == 0 || nrows == 0) {
      matrix = matrix(,0,0)
    } else { 
      matrix = matrix(,nrows,ncols)
      colNames = c()
      for (col in 1:ncols) {
        colNames = c(colNames, .jcall(stringMatrix, "S", "getColumnName", col))
        for (row in 1:nrows) {
          value = .jcall(stringMatrix, "S", "get", row, col)
          matrix[row,col] = .rdf.to.native(value)
        }
      }
      colnames(matrix) <- colNames
    }
    matrix
}

.rdf.to.native <- function(string) {
	result = string
	if (is.null(string)) {
		result = NA;
	} else if (is.na(string)) {
		# just return NA
	} else {
		c = strsplit(string, "\\^\\^")[[1]]
		if (length(c) == 2) {
			# possibly a RDF data type
			datatype = c[2]
			if (datatype == "http://www.w3.org/2001/XMLSchema#double") {
				result = as.numeric(c[1])
			} else if (datatype == "http://www.w3.org/2001/XMLSchema#float") {
				result = as.numeric(c[1])
			} else if (datatype == "http://www.w3.org/2001/XMLSchema#string") {
				result = c[1]
			}
		}
	}
	result
}
