% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/scpdsi.R
\name{pdsi}
\alias{pdsi}
\title{Calculate the (sc)PDSI}
\usage{
pdsi(P, PE, AWC = 100, start = NULL, end = NULL, cal_start = NULL,
  cal_end = NULL, sc = TRUE)
}
\arguments{
\item{P}{Monthly precipitation series without NA [mm]. Can be a time series.}

\item{PE}{Monthly potential evapotranspiration corresponding to the precipitation
series. Can be calculated by the Penman-Monteith or the Thonthwate
equation [mm].}

\item{AWC}{Available soil water capacity of the soil layer [mm]. Default 100 mm.}

\item{start}{Integer. Start year of the PDSI to be calculate default 1.}

\item{end}{Integer. End year of the PDSI to be calculate.}

\item{cal_start}{Integer. Start year of the calibrate period. Default is start year.}

\item{cal_end}{Integer. End year of the calibrate period. Default is end year.}

\item{sc}{Bool. Should use the self-calibrating procedure to calculate the climatical
coefficient (K2 and duration coefficients). If not it would use the default
parameters of Palmer (1965).}
}
\value{
This function return an object of class \code{pdsi}.

The object of class \code{pdsi} is a list containing the following components:

\itemize{
  \item call: the call to \code{pdsi} used to generate the object.
  \item X: time series of the X values, i.e. the Palmer Drought Severity Index (PDSI).
  \item inter.vars: An time series matrix containing the intermediate variables,
  including \code{P} (input precipitation), \code{PE} (input potential
  evapotranspiration), \code{PR} (potential recharge of soil moisture),
  \code{PRO} (potential runoff), \code{PL} (potential loss of soil moisture),
  \code{d} (water deficiencies), \code{Z} (water anomoly index, i.e. Z index),
  \code{Prob} (probability to end the wet or dry spell), \code{X1}, \code{X2} and
  \code{X3} (intermediate variables of calculating the X values).
  \item clim.coes: a matrix of the climate coefficients including \code{alpha},
  \code{beta}, \code{gamma}, \code{delta}, and \code{K1} coefficient for each month
  in a year.
  \item calib.coes: a matrix of the coefficients in the self-calibrating procedure
  of scPDSI, including \code{m}, \code{b} (slope and intercept of the
  duration-accumulated Z index plot), \code{p}, \code{q} (duration factors), and
  \code{K2} (ratio to adjust K coefficient) for wet and dry spell, respectively.
  Note that the P and PE would be convered from mm to inch in the calculation,
  therefore the units of \code{m}, \code{b} would also be inch correspondingly.

}
}
\description{
Calculating the monthly conventional Palmer Drought Severity Index
             (PDSI) and Self-calibrating PDSI (scPDSI) using the precipitation
             and potential evapotranspiration.
}
\details{
The Palmer Drought Severity Index (PDSI), proposed by Palmer (1965), is a
widely used drought indicator to quantify the long-term drought conditions, for
an area at a certain time. The PDSI is a semi-physical based drought index calculated
using the precipitation and potential evapotranspiration data, based on a simple
two-layer bucket water balance model. Conventionally, the constants to calculate
the PDSI were firstly empirically derived by using the meteorological records in
Kansas and Iowa in middle US with a semi-arid climate conditions, therefore the
conventional PDSI usually could not satisfactorily represent the drought conditions
for other areas around the world, which also makes spatial comparisons of PDSI values
difficult.

For this, Wells et al. (2004) proposed a self-calibrating Palmer Drought Severity
Index (scPDSI). The scPDSI could automatically adjust the empirical constants
in the PDSI computation with dynamically calculated values. Several works have
proved that the scPDSI performs better in spatially comparison than the conventional
PDSI. For more details please see the works of Wells et al. (2004).

This function could also calculate the conventional PDSI with revised constants.
Before the scPDSI appeared, the PDSI calculation has also been revised focusing
on the local climate characteristics in many area around the world. Those
constants could be reset by using the global options before calculating the PDSI
e.g.:

\preformatted{
options(PDSI.coe.K1.1 = 1.6)
}

And calculation in the PDSI would be:
\deqn{Ki0=coeK11 * lg(((PEi+Ri+ROi)/(Pi+Li)+coeK12)/Di)+coeK13}
\deqn{Ki=coeK2/(\sum Dj*Kj) * Ki}
\deqn{X[i]=p*X[i-1]+q*Z[i]}

\eqn{coeK11}, \eqn{coeK12}, \eqn{coeK13}, \eqn{coeK2}, \eqn{p}, and \eqn{q} are
corresponding to \code{PDSI.coe.K1.1}, \code{PDSI.coe.K1.2}, \code{PDSI.coe.K1.3},
\code{PDSI.coe.K2}, \code{PDSI.p}, and \code{PDSI.q}, respectively.

For example, in a national standard in China about meteorological drought level
(GB/T 20481-2017), the PDSI was revised by re-adjust the constants in the
calculation. To use the PDSI based on this standard should set the global
options of R as follows:

\preformatted{
options(PDSI.coe.K1.1 = 1.6)
options(PDSI.coe.K1.3 = 0.4)
options(PDSI.coe.K2 = 16.84)
options(PDSI.p = 0.755)
options(PDSI.q = 1/1.63)
}
}
\examples{
library(scPDSI)
data(Lubuge)

P <- Lubuge$P
PE <- Lubuge$PE
sc_pdsi <- pdsi(P, PE, start = 1960)
plot(sc_pdsi)

# Without self-calibrating.
ori_pdsi <- pdsi(P, PE, start = 1960, sc = FALSE)
plot(ori_pdsi)

# Without self-calibrating and use standards of
# mainland China. (GB/T 20481-2017)
options(PDSI.coe.K1.1 = 1.6)
options(PDSI.coe.K1.3 = 0.4)
options(PDSI.coe.K2 = 16.84)
options(PDSI.p = 0.755)
options(PDSI.q = 1/1.63)
gb_pdsi <- pdsi(P, PE, start = 1960, sc = FALSE)
plot(gb_pdsi)

}
\references{
Palmer W., 1965. Meteorological drought. U.s.department of Commerce
            Weather Bureau Research Paper.

            Wells, N., Goddard, S., Hayes, M. J., 2004. A Self-Calibrating Palmer
            Drought Severity Index. Journal of Climate, 17(12):2335-2351.
}

