% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Windham.R
\name{Windham}
\alias{Windham}
\title{Windham Robustification of Point Estimators for Exponential Family Distributions}
\usage{
Windham(
  Y,
  estimator,
  ldenfun,
  cW,
  ...,
  fpcontrol = list(Method = "Simple", ConvergenceMetricThreshold = 1e-10),
  paramvec_start = NULL,
  alternative_populationinverse = FALSE
)
}
\arguments{
\item{Y}{A matrix of measurements. Each row is a measurement, each component is a dimension of the measurement.}

\item{estimator}{A function that estimates parameters from weighted observations.
It must have arguments \code{Y} that is a matrix of measurements and \code{w} that are weights associated with each row of \code{Y}. If it accepts arguments \code{paramvec} or \code{paramvec_start} then these will be used to specify fixed elements of the parameter vector and the starting guess of the parameter vector, respectively. The estimated parameter vector, including any fixed elements, must be the returned object, or the first element of a returned list, or as the \code{paramvec} slot within the \code{est} slot of the returned object.}

\item{ldenfun}{A function that returns a vector of values proportional to the log-density for a matrix of observations \code{Y} and parameter vector \code{theta}.}

\item{cW}{A vector of robustness tuning constants. When computing the weight for an observation the parameter vector is multiplied element-wise with \code{cW}. For the PPI model, generate \code{cW} easily using \code{\link[=ppi_cW]{ppi_cW()}} and \code{\link[=ppi_cW_auto]{ppi_cW_auto()}}.}

\item{...}{Arguments passed to \code{estimator}.}

\item{fpcontrol}{A named list of control arguments to pass to \code{\link[FixedPoint:FixedPoint]{FixedPoint::FixedPoint()}} for the fixed point iteration.}

\item{paramvec_start}{Initially used to check the function \code{estimator}. If \code{estimator} accepts a \code{paramvec_start}, then the current estimate of the parameter vector is passed as \code{paramvec_start} to \code{estimator} in each iteration.}

\item{alternative_populationinverse}{The default is to use \code{\link[=Windham_populationinverse]{Windham_populationinverse()}}. If TRUE an alternative implementation in \code{\link[=Windham_populationinverse_alternative]{Windham_populationinverse_alternative()}} is used. So far we have not seen any difference between the results.}
}
\value{
A list:
\itemize{
\item \code{paramvec} the estimated parameter vector
\item \code{optim} information about the fixed point iterations and optimisation process. Including a slot \code{finalweights} for the weights in the final iteration.
}
}
\description{
Performs a generalisation of Windham's robustifying method \insertCite{windham1995ro}{scorematchingad} for exponential models with natural parameters that are a linear function of the parameters for estimation.
Estimators must solve estimating equations of the form
\deqn{\sum_{i = 1}^n U(z_i; \theta) = 0.}
The estimate is found iteratively through a fixed point method as suggested by \insertCite{windham1995ro;textual}{scorematchingad}.
}
\details{
For any family of models with density \eqn{f(z; \theta)}, Windham's method finds the parameter set \eqn{\hat\theta} such that the estimator applied to observations weighted by \eqn{f(z; \hat\theta)^c} returns an estimate that matches the theoretical effect of weighting the full population of the model.
When \eqn{f} is proportional to \eqn{\exp(\eta(\theta) \cdot T(z))} and \eqn{\eta(\theta)} is linear, these weights are equivalent to \eqn{f(z; c\hat\theta)} and the theoretical effect of the weighting on the full population is to scale the parameter vector \eqn{\theta} by \eqn{1+c}.

The function \code{Windham()} assumes that \eqn{f} is proportional to \eqn{\exp(\eta(\theta) \cdot T(z))} and \eqn{\eta(\theta)} is linear. It allows a generalisation where \eqn{c} is a vector so the weight for an observation \eqn{z} is \deqn{f(z; c \circ \theta),} where \eqn{\theta} is the parameter vector, \eqn{c} is a vector of tuning constants, and \eqn{\circ} is the element-wise product (Hadamard product).

The solution is found iteratively \insertCite{windham1995ro}{scorematchingad}.
Given a parameter set \eqn{\theta_n}, \code{Windham()} first computes weights \eqn{f(z; c \circ \theta_n)} for each observation \eqn{z}.
Then, a new parameter set \eqn{\tilde{\theta}_{n+1}} is estimated by \code{estimator} with the computed weights.
This new parameter set is element-wise-multiplied by the (element-wise) reciprocal of \eqn{1+c} to obtain an adjusted parameter set \eqn{\theta_{n+1}}.
The estimate returned by \code{Windham()} is the parameter set \eqn{\hat{\theta}} such that \eqn{\theta_n \approx \theta_{n+1}}.
}
\examples{
if (requireNamespace("movMF")){
  Y <- movMF::rmovMF(1000, 100 * c(1, 1) / sqrt(2))
} else {
  Y <- matrix(rnorm(1000 * 2, sd = 0.01), ncol = 2)
  Y <- Y / sqrt(rowSums(Y^2))
}
Windham(Y = Y,
   estimator = vMF,
   ldenfun = function(Y, theta){ #here theta is km
     return(drop(Y \%*\% theta))
   },
   cW = c(0.01, 0.01),
   method = "Mardia")
}
\seealso{
Other generic score matching tools: 
\code{\link{cppad_closed}()},
\code{\link{cppad_search}()}

Other Windham robustness functions: 
\code{\link{ppi_robust}()},
\code{\link{vMF_robust}()}
}
\concept{Windham robustness functions}
\concept{generic score matching tools}
