% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/grimmer-map-total-n.R
\name{grimmer_map_total_n}
\alias{grimmer_map_total_n}
\title{GRIMMER-testing with hypothetical group sizes}
\usage{
grimmer_map_total_n(
  data,
  x1 = NULL,
  x2 = NULL,
  sd1 = NULL,
  sd2 = NULL,
  dispersion = .dispersion,
  n_min = .n_min,
  n_max = .n_max,
  constant = .constant,
  constant_index = .constant_index,
  ...
)
}
\arguments{
\item{data}{Data frame with string columns \code{x1}, \code{x2}, \code{sd1}, and \code{sd2}, as
well as numeric column \code{n}. The first two are reported group means. \code{sd1}
and \code{sd2} are reported group SDs. \code{n} is the reported total sample size. It
is not very important whether a value is in \code{x1} or in \code{x2} because, after
the first round of tests, the function switches roles between \code{x1} and
\code{x2}, and reports the outcomes both ways. The same applies to \code{sd1} and
\code{sd2}. However, do make sure the \verb{x*} and \verb{sd*} values are paired
accurately, as reported.}

\item{x1, x2, sd1, sd2}{Optionally, specify these arguments as column names in
\code{data}.}

\item{dispersion}{Numeric. Steps up and down from half the \code{n} values.
Default is \code{0:5}, i.e., half \code{n} itself followed by five steps up and down.}

\item{n_min}{Numeric. Minimal group size. Default is 1.}

\item{n_max}{Numeric. Maximal group size. Default is \code{NULL}, i.e., no
maximum.}

\item{constant}{Optionally, add a length-2 vector or a list of length-2
vectors (such as a data frame with exactly two rows) to accompany the pairs
of dispersed values. Default is \code{NULL}, i.e., no constant values.}

\item{constant_index}{Integer (length 1). Index of \code{constant} or the first
\code{constant} column in the output tibble. If \code{NULL} (the default), \code{constant}
will go to the right of \code{n_change}.}

\item{...}{Arguments passed down to \code{grimmer_map()}.}
}
\value{
A tibble with these columns:
\itemize{
\item \code{x}, the group-wise reported input statistic, is repeated in row pairs.
\item \code{n} is dispersed from half the input \code{n}, with \code{n_change} tracking the
differences.
\item \code{both_consistent} flags scenarios where both reported \code{x} values are
consistent with the hypothetical \code{n} values.
\item \code{case} corresponds to the row numbers of the input data frame.
\item \code{dir} is \code{"forth"} in the first half of rows and \code{"back"} in the second
half. \code{"forth"} means that \code{x2} from the input is paired with the larger
dispersed \code{n}, whereas \code{"back"} means that \code{x1} is paired with the larger
dispersed \code{n}.
\item Other columns from \code{grimmer_map()} are preserved
}
}
\description{
When reporting group means, some published studies only report
the total sample size but no group sizes corresponding to each mean.
However, group sizes are crucial for GRIMMER-testing.

In the two-groups case, \code{grimmer_map_total_n()} helps in these ways:
\itemize{
\item It creates hypothetical group sizes. With an even total sample size, it
incrementally moves up and down from half the total sample size. For example,
with a total sample size of 40, it starts at 20, goes on to 19 and 21, then
to 18 and 22, etc. With odd sample sizes, it starts from the two integers
around half.
\item It GRIMMER-tests all of these values together with the group means.
\item It reports all the scenarios in which both "dispersed" hypothetical group
sizes are GRIMMER-consistent with the group means.
}

All of this works with one or more total sample sizes at a time. Call
\code{audit_total_n()} for summary statistics.
}
\section{Summaries with \code{audit_total_n()}}{
 You can call
\code{audit_total_n()} following up on \code{grimmer_map_total_n()}
to get a tibble with summary statistics. It will have these columns:
\itemize{
\item \code{x1}, \code{x2}, \code{sd1}, \code{sd2}, and \code{n} are the original inputs.
\item \code{hits_total} is the number of scenarios in which all of
\code{x1}, \code{x2}, \code{sd1}, and \code{sd2} are GRIMMER-consistent. It is the sum
of \code{hits_forth} and \code{hits_back} below.
\item \code{hits_forth} is the number of both-consistent cases that result
from pairing \code{x2} and \code{sd2} with the larger dispersed \code{n} value.
\item \code{hits_back} is the same, except \code{x1} and \code{sd1} are
paired with the larger dispersed \code{n} value.
\item \code{scenarios_total} is the total number of test scenarios,
whether or not both \code{x1} and \code{sd1} as well as \code{x2} and \code{sd2}
are GRIMMER-consistent.
\item \code{hit_rate} is the ratio of \code{hits_total} to \code{scenarios_total}.
}
}

\examples{
# Run `grimmer_map_total_n()` on data like these:
df <- tibble::tribble(
  ~x1,    ~x2,    ~sd1,   ~sd2,   ~n,
  "3.43", "5.28", "1.09", "2.12", 70,
  "2.97", "4.42", "0.43", "1.65", 65
)
df

grimmer_map_total_n(df)

# `audit_total_n()` summaries can be more important than
# the detailed results themselves.
# The `hits_total` column shows all scenarios in
# which both divergent `n` values are GRIMMER-consistent
# with the `x*` values when paired with them both ways:
df \%>\%
  grimmer_map_total_n() \%>\%
  audit_total_n()

# By default (`dispersion = 0:5`), the function goes
# five steps up and down from `n`. If this sequence
# gets longer, the number of hits tends to increase:
df \%>\%
  grimmer_map_total_n(dispersion = 0:10) \%>\%
  audit_total_n()
}
\references{
Bauer, P. J., & Francis, G. (2021). Expression of Concern: Is It
Light or Dark? Recalling Moral Behavior Changes Perception of Brightness.
\emph{Psychological Science}, 32(12), 2042–2043.
https://journals.sagepub.com/doi/10.1177/09567976211058727

Allard, A. (2018). Analytic-GRIMMER: a new way of testing the
possibility of standard deviations.
https://aurelienallard.netlify.app/post/anaytic-grimmer-possibility-standard-deviations/

Bauer, P. J., & Francis, G. (2021). Expression of Concern: Is It Light or
Dark? Recalling Moral Behavior Changes Perception of Brightness.
\emph{Psychological Science}, 32(12), 2042–2043.
https://journals.sagepub.com/doi/10.1177/09567976211058727
}
\seealso{
\code{function_map_total_n()}, which created the present function using
\code{grimmer_map()}.
}
