% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ipf.R
\name{ipf}
\alias{ipf}
\title{Adjustment of marginal distributions using iterative proportional fitting}
\usage{
ipf(source, target, group, unit, weight = NULL, max_iterations = 100,
  precision = 0.001)
}
\arguments{
\item{source}{A "source" data frame. The marginals of this
dataset are adjusted to the marginals of \code{target}.}

\item{target}{A "target" data frame. The function returns a dataset
where the marginal distributions of \code{group} and \code{unit} categories
are approximated by those of \code{target}.}

\item{group}{A categorical variable or a vector of variables
contained in \code{source} and \code{target}. Defines the first distribution
for adjustment.}

\item{unit}{A categorical variable or a vector of variables
contained in \code{source} and \code{target}. Defines the second distribution
for adjustment.}

\item{weight}{Numeric. (Default \code{NULL})}

\item{max_iterations}{Maximum number of iterations used for the IPF algorithm.}

\item{precision}{Convergence criterion for the IPF algorithm. In every iteration,
the ratio of the source and target marginals are calculated for every category of
\code{group} and \code{unit}. The algorithm converges when all ratios are smaller
than \code{1 + precision}.}
}
\value{
Returns a data frame that retains
  the association structure of \code{source} while approximating
  the marginal distributions for \code{group} and \code{unit} of \code{target}.
  The dataset identifies each combination of \code{group} and \code{unit},
  and categories that only occur in either \code{source} or \code{target} are dropped.
  The adjusted frequency of each combination is given by the column \code{n},
  while \code{n_target} and \code{n_source} contain the zero-adjusted frequencies
  in the target and source dataset, respectively.
}
\description{
Adjusts the marginal distributions for \code{group} and \code{unit}
in \code{source} to the respective marginal distributions in \code{target}, using the iterative
proportional fitting algorithm (IPF).
}
\details{
The algorithm works by scaling
the marginal distribution of \code{group} in the \code{source} data frame towards the
marginal distribution of \code{target}; then repeating this process for \code{unit}. The
algorithm then keeps alternating between \code{group} and \code{unit} until the marginals
of the adjusted data frame are within the allowed precision. This results in a dataset that
retains the association structure of \code{source} while approximating
the marginal distribution of \code{target}. If the number of \code{unit} and
\code{group} categories is different in \code{source} and \code{target}, the data frame returns
the combination of \code{unit} and \code{group} categories that occur in both datasets.
Zero values are replaced by a small, non-zero number (1e-4).
Note that the values returned sum to the observations of the source data frame, not the
target data frame. This is different from other IPF implementations, but ensures that the IPF
does not change the number of observations.
}
\examples{
# adjusts the marginals of group and unit categories so that
# schools00 has similar marginals as schools05
adj <- ipf(schools00, schools05, "race", "school", weight = "n")

# check that the new "race" marginals are similar to the target marginals
# (the same could be done for schools)
aggregate(adj$n, list(adj$race), sum)
aggregate(adj$n_target, list(adj$race), sum)

# note that the adjusted dataset contains fewer
# schools than either the source or the target dataset,
# because the marginals are only defined for the overlap
# of schools
length(unique(schools00$school))
length(unique(schools05$school))
length(unique(adj$school))
}
\references{
W. E. Deming and F. F. Stephan. 1940.
  "On a Least Squares Adjustment of a Sampled Frequency Table
  When the Expected Marginal Totals are Known".
  Annals of Mathematical Statistics. 11 (4): 427–444.

 T. Karmel and M. Maclachlan. 1988.
  "Occupational Sex Segregation — Increasing or Decreasing?" Economic Record 64: 187-195.
}
