% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/standardizedSolution_boot_ci.R
\name{standardizedSolution_boot_ci}
\alias{standardizedSolution_boot_ci}
\alias{store_boot_est_std}
\alias{get_boot_est_std}
\title{Bootstrap CIs for Standardized
Solution}
\usage{
standardizedSolution_boot_ci(
  object,
  level = 0.95,
  type = "std.all",
  save_boot_est_std = TRUE,
  force_run = FALSE,
  boot_delta_ratio = FALSE,
  ...
)

store_boot_est_std(object, type = "std.all", force_run = FALSE, ...)

get_boot_est_std(object)
}
\arguments{
\item{object}{A \linkS4class{lavaan}
object, fitted with 'se = "boot"'.}

\item{level}{The level of confidence
of the confidence intervals. Default
is .95.}

\item{type}{The type of standard
estimates. The same argument of
\code{\link[lavaan:standardizedSolution]{lavaan::standardizedSolution()}},
and support all values supported by
\code{\link[lavaan:standardizedSolution]{lavaan::standardizedSolution()}}.
Default is \code{"std.all"}.}

\item{save_boot_est_std}{Whether the
bootstrap estimates of the
standardized solution are saved. If
saved, they will be stored in the
attribute \code{boot_est_std}. Default is
\code{TRUE}.}

\item{force_run}{If \code{TRUE}, will skip
checks and run models without
checking the estimates. For internal
use. Default is \code{FALSE}.}

\item{boot_delta_ratio}{The ratio of
(a) the distance of the bootstrap
confidence limit from the point
estimate to (b) the distance of the
delta-method limit from the point
estimate. Default is \code{FALSE}.}

\item{...}{Other arguments to be
passed to
\code{\link[lavaan:standardizedSolution]{lavaan::standardizedSolution()}}.}
}
\value{
The output of
\code{\link[lavaan:standardizedSolution]{lavaan::standardizedSolution()}},
with bootstrap confidence intervals
appended to the right, with class
set to \code{std_solution_boot} (since
version 0.1.8.4). It has
a print method
(\code{\link[=print.std_solution_boot]{print.std_solution_boot()}}) that
can be used to print the standardized
solution in a format similar to
that of the printout of
the \code{\link[=summary]{summary()}} of a \link[lavaan:lavaan-class]{lavaan::lavaan} object.

\code{\link[=store_boot_est_std]{store_boot_est_std()}} returns
the fit object set to
\code{object}, with the bootstrap values
of standardized solution in the
bootstrap samples, as a matrix,
stored in the
slot \code{external} under the name
\code{shh_boot_est_std}.

\code{\link[=get_boot_est_std]{get_boot_est_std()}} returns a matrix
of the stored bootstrap estimates
of standardized solution. If none is
stored, \code{NULL} is returned.

\code{\link[=store_boot_est_std]{store_boot_est_std()}} is usually used
with diagnostic functions such
as \code{\link[=plot_boot]{plot_boot()}}.
}
\description{
Functions for forming
bootstrap confidence intervals
for the standardized solution.
}
\details{
\code{\link[=standardizedSolution_boot_ci]{standardizedSolution_boot_ci()}}
receives a
\link[lavaan:lavaan-class]{lavaan::lavaan} object fitted
with bootstrapping standard errors
requested and forms the confidence
intervals for the standardized
solution.

It works by calling
\code{\link[lavaan:standardizedSolution]{lavaan::standardizedSolution()}}
with the bootstrap estimates
of free parameters in each bootstrap sample
to compute the standardized estimates
in each sample.

A more reliable way is to use
function like
\code{\link[lavaan:bootstrap]{lavaan::bootstrapLavaan()}}.
Nevertheless, this simple function is
good enough for some simple scenarios,
and does not require repeating
the bootstrapping step.

\code{\link[=store_boot_est_std]{store_boot_est_std()}} computes the
standardized solution for each bootstrap
sample, stores them the
\link[lavaan:lavaan-class]{lavaan::lavaan} object, and
returns it. These estimates can be used
by other functions, such as \code{\link[=plot_boot]{plot_boot()}},
to examine the
estimates, without the need
to repeat the computation.

\code{\link[=get_boot_est_std]{get_boot_est_std()}} retrieves
the bootstrap estimates of the
standardized solution stored by
\code{\link[=store_boot_est_std]{store_boot_est_std()}}.
}
\examples{

library(lavaan)
set.seed(5478374)
n <- 50
x <- runif(n) - .5
m <- .40 * x + rnorm(n, 0, sqrt(1 - .40))
y <- .30 * m + rnorm(n, 0, sqrt(1 - .30))
dat <- data.frame(x = x, y = y, m = m)
model <-
'
m ~ a*x
y ~ b*m
ab := a*b
'

# Should set bootstrap to at least 2000 in real studies
fit <- sem(model, data = dat, fixed.x = FALSE,
           se = "boot",
           bootstrap = 100)
summary(fit)

std <- standardizedSolution_boot_ci(fit)
std

# Print in a friendly format with only standardized solution
print(std, output = "text")

# Print in a friendly format with both unstandardized
# and standardized solution
print(std, output = "text", standardized_only = FALSE)


# store_boot_est_std() is usually used with plot_boot()
# First, store the bootstrap estimates of the
# standardized solution
fit_with_boot_std <- store_boot_est_std(fit)
# Second, plot the distribution of the bootstrap estimates of
# standardized 'ab'
plot_boot(fit_with_boot_std, "ab", standardized = TRUE)

}
\seealso{
\code{\link[lavaan:standardizedSolution]{lavaan::standardizedSolution()}}, \code{\link[=plot_boot]{plot_boot()}}
}
\author{
Shu Fai Cheung
\url{https://orcid.org/0000-0002-9871-9448}.
Originally proposed in an issue at GitHub
\url{https://github.com/simsem/semTools/issues/101#issue-1021974657},
inspired by a discussion at
the Google group for lavaan
\url{https://groups.google.com/g/lavaan/c/qQBXSz5cd0o/m/R8YT5HxNAgAJ}.
\code{\link[boot:boot.ci]{boot::boot.ci()}} is used to form the
percentile confidence intervals in
this version.
}
