% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tree_influ_phyglm.R
\name{tree_influ_phyglm}
\alias{tree_influ_phyglm}
\title{Interaction between phylogenetic uncertainty and influential species detection - Phylogenetic Logistic Regression}
\usage{
tree_influ_phyglm(formula, data, phy, n.tree = 2, cutoff = 2, btol = 50,
  track = TRUE, ...)
}
\arguments{
\item{formula}{The model formula}

\item{data}{Data frame containing species traits with row names matching tips
in \code{phy}.}

\item{phy}{A phylogeny (class 'phylo') matching \code{data}.}

\item{n.tree}{Number of times to repeat the analysis with n different trees picked 
randomly in the multiPhylo file.}

\item{cutoff}{The cutoff value used to identify for influential species
(see Details)}

\item{btol}{Bound on searching space. For details see \code{phyloglm}}

\item{track}{Print a report tracking function progress (default = TRUE)}

\item{...}{Further arguments to be passed to \code{phyloglm}}
}
\value{
The function \code{influ_phyglm} returns a list with the following
components:

\code{cutoff}: The value selected for \code{cutoff}

\code{formula}: The formula

\code{full.model.estimates}: Coefficients, aic and the optimised
value of the phylogenetic parameter (i.e. \code{alpha}) for the full model
without deleted species.

\code{influential_species}: List of influential species, both
based on standardised difference in interecept and in the slope of the
regression. Species are ordered from most influential to less influential and
only include species with a standardised difference > \code{cutoff}.

\code{sensi.estimates}: A data frame with all simulation
estimates. Each row represents a deleted clade for a given random tree. Columns report the calculated
regression intercept (\code{intercept}), difference between simulation
intercept and full model intercept (\code{DIFintercept}), the standardised
difference (\code{sDIFintercept}), the percentage of change in intercept compared
to the full model (\code{intercept.perc}) and intercept p-value
(\code{pval.intercept}). All these parameters are also reported for the regression
slope (\code{DIFestimate} etc.). Additionally, model aic value (\code{AIC}) and
the optimised value (\code{optpar}) of the phylogenetic parameter
(i.e. \code{alpha}) are reported.

\code{data}: Original full dataset.

\code{errors}: Species where deletion resulted in errors.
}
\description{
Performs leave-one-out deletion analysis for phylogenetic logistic regression,
and detects influential species while evaluating uncertainty in trees topology.
}
\details{
This function sequentially removes one species at a time, fits a phylogenetic
logistic regression model using \code{\link[phylolm]{phyloglm}}, stores the
results and detects influential species. It repeats this operation using n trees, 
randomly picked in a multiPhylo file.

Currently only logistic regression using the "logistic_MPLE"-method from
\code{phyloglm} is implemented.

\code{influ_phyglm} detects influential species based on the standardised
difference in intercept and/or slope when removing a given species compared
to the full model including all species. Species with a standardised difference
above the value of \code{cutoff} are identified as influential. The default
value for the cutoff is 2 standardised differences change.

Currently, this function can only implement simple logistic models (i.e. \eqn{trait~
predictor}). In the future we will implement more complex models.

Output can be visualised using \code{sensi_plot}.
}
\examples{
\dontrun{
# Simulate Data:
set.seed(6987)
mphy = rmtree(100, N = 30)
x = rTrait(n=1,phy=mphy[[1]])
X = cbind(rep(1,100),x)
y = rbinTrait(n=1,phy=mphy[[1]], beta=c(-1,0.5), alpha=.7 ,X=X)
dat = data.frame(y, x)
# Run sensitivity analysis:
tree_influ <- tree_influ_phyglm(y ~ x, data = dat, phy = mphy, n.tree = 5)
summary(tree_influ)
sensi_plot(tree_influ)
sensi_plot(tree_influ, graphs = 1)
sensi_plot(tree_influ, graphs = 2)
}
\dontshow{
set.seed(6987)
mphy = rmtree(25, N = 30)
x = rTrait(n=1,phy=mphy[[1]])
X = cbind(rep(1,25),x)
y = rbinTrait(n=1,phy=mphy[[1]], beta=c(-1,0.5), alpha=.7 ,X=X)
dat = data.frame(y, x)
tree_influ <- tree_influ_phyglm(y ~ x, data = dat, phy = mphy, n.tree = 2)
}
}
\references{
Ho, L. S. T. and Ane, C. 2014. "A linear-time algorithm for 
Gaussian and non-Gaussian trait evolution models". Systematic Biology 63(3):397-408.
}
\seealso{
\code{\link[phylolm]{phyloglm}}, \code{\link{tree_phyglm}},
\code{\link{influ_phyglm}}, \code{\link{tree_influ_phyglm}}, \code{\link{sensi_plot}}
}
\author{
Gustavo Paterno, Caterina Penone & Gijsbert D.A. Werner
}
