% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/influ_continuous.R
\name{influ_continuous}
\alias{influ_continuous}
\title{Influential Species Detection - Trait Evolution Continous Characters}
\usage{
influ_continuous(data, phy, model, bounds = list(), cutoff = 2,
  n.cores = NULL, track = TRUE, ...)
}
\arguments{
\item{data}{Data vector for a single continuous trait, with names matching tips in \code{phy}.}

\item{phy}{A phylogeny (class 'phylo') matching \code{data}.}

\item{model}{The evolutionary model (see Details).}

\item{bounds}{settings to constrain parameter estimates. See \code{\link[geiger]{fitContinuous}}}

\item{cutoff}{The cut-off parameter for influential species (see Details).}

\item{n.cores}{number of cores to use. If 'NULL', number of cores is detected.}

\item{track}{Print a report tracking function progress (default = TRUE)}

\item{...}{Further arguments to be passed to \code{\link[geiger]{fitContinuous}}}
}
\value{
The function \code{tree_discrete} returns a list with the following
components:

\code{call}: The function call

\code{cutoff}: The value selected for \code{cutoff}

\code{data}: The original full data vector

\code{optpar}: Transformation parameter used (e.g. \code{lambda}, \code{kappa} etc.)

\code{full.model.estimates}: Parameter estimates (rate of evolution \code{sigsq}
and where applicable \code{optpar}), root state \code{z0}, 
AICc for the full model without deleted species.

\code{influential_species}: List of influential species, based on standardised 
difference in estimates for sigsq and optpar. Species are ordered from most influential to 
less influential and only include species with a standardised difference > \code{cutoff}.

\code{sensi.estimates}: Parameter estimates (sigsq and optpar),(percentual) difference 
in parameter estimate compared to the full model (DIFsigsq, sigsq.perc,sDIFsigsq, 
DIFoptpar, optpar.perc,sDIFoptpar), 
AICc and z0 for each repeat with a species removed.
}
\description{
Fits models for trait evolution of continuous characters, 
detecting influential species.
}
\details{
This function sequentially removes one species at a time,  
fits different models of continuous character evolution using \code{\link[geiger]{fitContinuous}}, 
stores the results and calculates the effects on model parameters.

\code{influ_continuous} detects influential species based on the standardised
difference in the rate parameter \code{sigsq} and the optimisation parameter \code{optpar} 
(e.g. lamda, kappa, alpha, depending on which \code{model} is set), when removing 
a given species compared to the full model including all species. 
Species with a standardised difference above the value of 
\code{cutoff} are identified as influential. 

Different evolutionary models from \code{fitContinuous} can be used, i.e. \code{BM},\code{OU},
\code{EB}, \code{trend}, \code{lambda}, \code{kappa}, \code{delta} and \code{drift}.

See \code{\link[geiger]{fitContinuous}} for more details on evolutionary models.
}
\examples{
\dontrun{
#Load data:
data("primates")
#Model trait evolution accounting for influential species
adultMass<-primates$data$adultMass
names(adultMass)<-rownames(primates$data)
influ_cont<-influ_continuous(data = adultMass,phy = primates$phy[[1]],
model = "OU",cutoff = 2,n.cores = 2,track = TRUE)
#Print summary statistics
summary(influ_cont)
sensi_plot(influ_cont)
sensi_plot(influ_cont,graphs="sigsq")
#' sensi_plot(influ_cont,graphs="optpar")
#Use a different evolutionary model or cutoff 
influ_cont2<-influ_continuous(data = adultMass,phy = primates$phy[[1]],
model = "lambda",cutoff = 1.2,n.cores = 2,track = TRUE)
summary(influ_cont2)
sensi_plot(influ_cont2)
influ_cont3<-influ_continuous(data = adultMass,phy = primates$phy[[1]],
model = "BM",cutoff = 2,n.cores = 2,track = TRUE)
summary(influ_cont3)
}
}
\references{
Yang Z. 2006. Computational Molecular Evolution. Oxford University Press: Oxford. 

Harmon Luke J, Jason T Weir, Chad D Brock, Richard E Glor, and Wendell Challenger. 2008.
GEIGER: investigating evolutionary radiations. Bioinformatics 24:129-131.
}
\seealso{
\code{\link[geiger]{fitContinuous}}
}
\author{
Gijsbert Werner & Gustavo Paterno
}
