% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/influ_Discrete.R
\name{influ_discrete}
\alias{influ_discrete}
\title{Influential Species Detection - Trait Evolution Discrete Characters}
\usage{
influ_discrete(
  data,
  phy,
  model,
  transform = "none",
  bounds = list(),
  cutoff = 2,
  n.cores = NULL,
  track = TRUE,
  ...
)
}
\arguments{
\item{data}{Data vector for a single binary trait, with names matching tips in \code{phy}.}

\item{phy}{A phylogeny (class 'phylo') matching \code{data}.}

\item{model}{The Mkn model to use (see Details).}

\item{transform}{The evolutionary model to transform the tree (see Details). Default is \code{none}.}

\item{bounds}{settings to constrain parameter estimates. See \code{\link[geiger]{fitDiscrete}}}

\item{cutoff}{The cut-off parameter for influential species (see Details).}

\item{n.cores}{number of cores to use. If 'NULL', number of cores is detected.}

\item{track}{Print a report tracking function progress (default = TRUE)}

\item{...}{Further arguments to be passed to \code{\link[geiger]{fitDiscrete}}}
}
\value{
The function \code{tree_discrete} returns a list with the following
components:

\code{call}: The function call

\code{cutoff}: The value selected for \code{cutoff}

\code{data}: The original full data vector

\code{optpar}: Transformation parameter used (e.g. \code{lambda}, \code{kappa} etc.)

\code{full.model.estimates}: Parameter estimates (transition rates q12 and q21), 
AICc and the optimised value of the phylogenetic transformation parameter (e.g. \code{lambda}) 
for the full model.

\code{influential_species}: List of influential species, based on standardised 
difference in estimates for q12 and q21. Species are ordered from most influential to 
less influential and only include species with a standardised difference > \code{cutoff}.

\code{sensi.estimates}: Parameter estimates (transition rates q12 and q21),,(percentual) difference 
in parameter estimate compared to the full model (DIFq12, sigsq.q12,sDIFq12, DIFq21, optpar.q21,sDIFq21),  
AICc and the optimised value of the phylogenetic transformation parameter (e.g. \code{lambda}) 
for each analysis with a species deleted.
}
\description{
Fits models for trait evolution of discrete (binary) characters, 
detecting influential species.
}
\details{
This function sequentially removes one species at a time,  
fits a model of discrete character evolution using \code{\link[geiger]{fitDiscrete}}, 
stores the results and calculates the effects on model parameters. Currently, only 
binary discrete traits are supported.

\code{influ_discrete} detects influential species based on the standardised
difference in q12 or q21 when removing a given species compared
to the full model including all species. Species with a standardised difference
above the value of \code{cutoff} are identified as influential. 

Different character model from \code{fitDiscrete} can be used, including \code{ER} (equal-rates), 
\code{SYM} (symmetric), \code{ARD} (all-rates-different) and \code{meristic} (stepwise fashion). 

Different transformations to the phylogenetic tree from \code{fitDiscrete} can be used, i.e. \code{none},
\code{EB}, \code{lambda}, \code{kappa} and\code{delta}.

See \code{\link[geiger]{fitDiscrete}} for more details on character models and tree transformations.
}
\examples{
\dontrun{
#Load data:
data("primates")
#Create a binary trait factor 
adultMass_binary<-ifelse(primates$data$adultMass > 7350, "big", "small")
adultMass_binary<-as.factor(as.factor(adultMass_binary))
names(adultMass_binary)<-rownames(primates$data)
#Model trait evolution accounting for influential species
influ_binary<-influ_discrete(data = adultMass_binary,phy = primates$phy[[1]],
model = "SYM",transform = "none",cutoff = 2,n.cores = 2,track = TRUE)
#Print summary statistics
summary(influ_binary)
sensi_plot(influ_binary) #q12 and q21 are, as expected, exactly the same in symmetrical model. 
#Use a different evolutionary model. 
influ_binary2<-influ_discrete(data = adultMass_binary,phy = primates$phy[[1]],
model = "SYM",transform = "delta",n.cores = 2,track = TRUE)
summary(influ_binary2)
sensi_plot(influ_binary2)
#Or change the cutoff and transformation
influ_binary3<-influ_discrete(data = adultMass_binary,phy = primates$phy[[1]],
model = "ARD",transform = "none",cutoff = 1.2,n.cores = 2,track = TRUE)
summary(influ_binary3)
sensi_plot(influ_binary3) 
}
}
\references{
Paterno, G. B., Penone, C. Werner, G. D. A. 
\href{http://doi.wiley.com/10.1111/2041-210X.12990}{sensiPhy: 
An r-package for sensitivity analysis in phylogenetic 
comparative methods.} Methods in Ecology and Evolution 
2018, 9(6):1461-1467.  

Yang Z. 2006. Computational Molecular Evolution. Oxford University Press: Oxford. 

Harmon Luke J, Jason T Weir, Chad D Brock, Richard E Glor, and Wendell Challenger. 2008.
GEIGER: investigating evolutionary radiations. Bioinformatics 24:129-131.
}
\seealso{
\code{\link[geiger]{fitDiscrete}}
}
\author{
Gijsbert Werner & Gustavo Paterno
}
