% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sentiment_engines.R
\name{compute_sentiment}
\alias{compute_sentiment}
\title{Compute document-level sentiment across features and lexicons}
\usage{
compute_sentiment(x, lexicons, how = "proportional", nCore = 1,
  dfm = NULL)
}
\arguments{
\item{x}{either a \code{sentocorpus} object created with \code{\link{sento_corpus}}, a \pkg{quanteda}
\code{\link[quanteda]{corpus}} object, or a \code{character} vector. The latter two do not incorporate a
date dimension. In case of a \code{\link[quanteda]{corpus}} object, the \code{numeric} columns from the
\code{\link[quanteda]{docvars}} are considered as features over which sentiment will be computed. In
case of a \code{character} vector, sentiment is only computed across lexicons.}

\item{lexicons}{output from a \code{\link{setup_lexicons}} call.}

\item{how}{a single \code{character} vector defining how aggregation within documents should be performed. For currently
available options on how aggregation can occur, see \code{\link{get_hows}()$words}.}

\item{nCore}{a single \code{numeric} at least equal to 1 to indicate the number of cores to use for a parallel sentiment
computation. We use the \code{\%dopar\%} construct from the \pkg{foreach} package. By default, \code{nCore = 1}, which
implies no parallelization.}

\item{dfm}{(optional) an output from a \pkg{quanteda} \code{\link[quanteda]{dfm}} call, such that users can specify their
own tokenisation scheme (via \code{\link[quanteda]{tokens}}) as well as other parameters related to the construction of
a document-feature matrix (dfm). Make sure the document-feature matrix is constructed from the texts in the
\code{sentocorpus} object, otherwise, results will be spurious or errors may occur. Note that valence shifters will
not be integrated into the features of a user-provided dfm.}
}
\value{
A \code{list} containing:
\item{corpus}{the supplied \code{x} object, transformed into a \code{\link[quanteda]{corpus}} if a \code{character} vector.}
\item{sentiment}{the sentiment scores \code{data.table} with a \code{"word_count"} column and all
lexicon--feature sentiment scores columns.}
\item{features}{a \code{character} vector of the different features.}
\item{lexicons}{a \code{character} vector of the different lexicons used.}
\item{howWithin}{the supplied \code{how} argument.}

The last three elements are only present if \code{x} is a \code{sentocorpus} object. In that case, the
\code{"sentiment"} \code{data.table} also has a \code{"date"} column, meaning it can be used for further
aggregation into sentiment time series with the \code{\link{perform_agg}} function.
}
\description{
Given a corpus of texts, computes sentiment per document using the bag-of-words approach
based on the lexicons provided and a choice of aggregation across words per document. Relies partly on the
\pkg{quanteda} package. The scores computed are net sentiment (sum of positive minus sum of negative scores).
}
\details{
For a separate calculation of positive (resp. negative) sentiment, one has to provide distinct positive (resp. negative)
lexicons. This can be done using the \code{do.split} option in the \code{\link{setup_lexicons}} function, which splits out
the lexicons into a positive and a negative polarity counterpart. \code{NA}s are converted to 0, under the assumption that
this is equivalent to no sentiment. By default, if the \code{dfm} argument is left unspecified, a document-feature matrix
(dfm) is created based on a tokenisation that removes punctuation, numbers, symbols and separators, but does not remove
stopwords. The number of words for each document is computed based on that same tokenisation. All tokens are converted to
lowercase, in line with what the \code{\link{setup_lexicons}} function does for the lexicons and valence shifters.
}
\examples{
data("usnews", package = "sentometrics")
data("list_lexicons", package = "sentometrics")
data("list_valence_shifters", package = "sentometrics")

l1 <- list_lexicons[c("LM_en", "HENRY_en")]
l2 <- setup_lexicons(list_lexicons[c("LM_en", "HENRY_en")], list_valence_shifters[["en"]])

# from a sentocorpus object
corpus <- sento_corpus(corpusdf = usnews)
corpusSample <- quanteda::corpus_sample(corpus, size = 200)
sent <- compute_sentiment(corpusSample, l1, how = "counts")

# from a character vector
sent <- compute_sentiment(usnews[["texts"]][1:200], l1, how = "counts")

\dontrun{
# from a corpus object, parallelized
corpusQ <- quanteda::corpus(usnews, text_field = "texts")
sent <- compute_sentiment(corpusQ, l2, how = "counts", nCore = 2)}

\dontrun{
# using a user-supplied dfm with default settings
tok <- quanteda::tokens_tolower(quanteda::tokens(corpus))
dfm <- quanteda::dfm(tok, verbose = FALSE)
sent <- compute_sentiment(corpus, l1, how = "counts", dfm = dfm)}

}
\seealso{
\code{\link[quanteda]{dfm}}, \code{\link[quanteda]{tokens}}
}
\author{
Samuel Borms
}
