% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/genthin.R
\name{thin_all}
\alias{thin_all}
\title{Binomial thinning for altering read-depth.}
\usage{
thin_all(mat, thinlog2)
}
\arguments{
\item{mat}{A numeric matrix of RNA-seq counts. The rows index the genes and
the columns index the samples.}

\item{thinlog2}{A numeric scalar. This is the amount to shrink each count
in \code{mat} (on the log2-scale).  For
example, a value of 0 means that we do not thin, a value of 1 means
that we thin by a factor of 2, a value of 2 means we thin by a factor
of 4, etc.}
}
\value{
A list-like S3 object of class \code{ThinData}.
Components include some or all of the following:
\describe{
  \item{\code{mat}}{The modified matrix of counts.}
  \item{\code{designmat}}{The design matrix of variables used to simulate
      signal. This is made by column-binding \code{design_fixed} and the
      permuted version of \code{design_perm}.}
  \item{\code{coefmat}}{A matrix of coefficients corresponding to
      \code{designmat}.}
  \item{\code{design_obs}}{Additional variables that should be included in
      your design matrix in downstream fittings. This is made by
      column-binding the vector of 1's with \code{design_obs}.}
  \item{\code{sv}}{A matrix of estimated surrogate variables. In simulation
      studies you would probably leave this out and estimate your own
      surrogate variables.}
  \item{\code{cormat}}{A matrix of target correlations between the
      surrogate variables and the permuted variables in the design matrix.
      This might be different from the \code{target_cor} you input because
      we pass it through \code{\link{fix_cor}} to ensure
      positive semi-definiteness of the resulting covariance matrix.}
  \item{\code{matching_var}}{A matrix of simulated variables used to
      permute \code{design_perm} if the \code{target_cor} is not
      \code{NULL}.}
}
}
\description{
Given a matrix of real RNA-seq counts, this function will apply a
thinning factor uniformly to every count in this matrix. This uniformly
lowers the read-depth for the entire dataset. The thinning factor should
be provided on the log2-scale. This is a specific application of the
binomial thinning approach in \code{\link{thin_diff}}. Though this particular
form of thinning was used by Robinson and Storey (2014) in the context
of deriving read-depth suggestions.
}
\examples{
## Generate count data and set thinning factor
## In practice, you would obtain mat from a real dataset, not simulate it.
set.seed(1)
n <- 10
p <- 1000
lambda <- 1000
mat <- matrix(lambda, ncol = n, nrow = p)
thinlog2 <- 1

## Thin read-depths
thout <- thin_all(mat = mat, thinlog2 = thinlog2)

## Compare empirical and theoretical proportions
mean(thout$mat) / lambda
2 ^ -thinlog2

}
\references{
\itemize{
  \item{Robinson, David G., and John D. Storey. "subSeq: determining appropriate sequencing depth through efficient read subsampling." Bioinformatics 30, no. 23 (2014): 3424-3426.}
}
}
\seealso{
\describe{
  \item{\code{\link{select_counts}}}{For subsampling the rows and columns
      of your real RNA-seq count matrix prior to applying binomial thinning.}
  \item{\code{\link{thin_diff}}}{For the more general thinning approach.}
  \item{\code{\link{thin_lib}}}{For thinning sample-wise.}
  \item{\code{\link{thin_gene}}}{For thinning gene-wise.}
  \item{\code{\link{ThinDataToSummarizedExperiment}}}{For converting a
      ThinData object to a SummarizedExperiment object.}
  \item{\code{\link{ThinDataToDESeqDataSet}}}{For converting a
      ThinData object to a DESeqDataSet object.}
}
}
\author{
David Gerard
}
