% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/genthin.R
\name{thin_gene}
\alias{thin_gene}
\title{Binomial thinning for altering total gene expression levels}
\usage{
thin_gene(mat, thinlog2, relative = FALSE)
}
\arguments{
\item{mat}{A numeric matrix of RNA-seq counts. The rows index the genes and
the columns index the samples.}

\item{thinlog2}{A vector of numerics. Element i is the amount to thin
(on the log2 scale) for gene i. For
example, a value of 0 means that we do not thin, a value of 1 means
that we thin by a factor of 2, a value of 2 means we thin by a factor
of 4, etc.}

\item{relative}{A logical. Should we apply relative thinning (\code{TRUE})
or absolute thinning (\code{FALSE}). Only experts should change
the default.}
}
\value{
A list-like S3 object of class \code{ThinData}.
Components include some or all of the following:
\describe{
  \item{\code{mat}}{The modified matrix of counts.}
  \item{\code{designmat}}{The design matrix of variables used to simulate
      signal. This is made by column-binding \code{design_fixed} and the
      permuted version of \code{design_perm}.}
  \item{\code{coefmat}}{A matrix of coefficients corresponding to
      \code{designmat}.}
  \item{\code{design_obs}}{Additional variables that should be included in
      your design matrix in downstream fittings. This is made by
      column-binding the vector of 1's with \code{design_obs}.}
  \item{\code{sv}}{A matrix of estimated surrogate variables. In simulation
      studies you would probably leave this out and estimate your own
      surrogate variables.}
  \item{\code{cormat}}{A matrix of target correlations between the
      surrogate variables and the permuted variables in the design matrix.
      This might be different from the \code{target_cor} you input because
      we pass it through \code{\link{fix_cor}} to ensure
      positive semi-definiteness of the resulting covariance matrix.}
  \item{\code{matching_var}}{A matrix of simulated variables used to
      permute \code{design_perm} if the \code{target_cor} is not
      \code{NULL}.}
}
}
\description{
Given a matrix of real RNA-seq counts, this function will apply a
separate, user-provided thinning factor to each gene. This uniformly
lowers the counts for all samples in a gene. The thinning factor
should be provided on the log2-scale. This is a specific application
of the binomial thinning approach in \code{\link{thin_diff}}.
}
\examples{
## Generate count data and thinning factors
## In practice, you would obtain mat from a real dataset, not simulate it.
set.seed(1)
n <- 10
p <- 1000
lambda <- 1000
mat <- matrix(lambda, ncol = n, nrow = p)
thinlog2 <- rexp(n = p, rate = 1)

## Thin total gene expressions
thout <- thin_gene(mat = mat, thinlog2 = thinlog2)

## Compare empirical thinning proportions to specified thinning proportions
empirical_propvec <- rowMeans(thout$mat) / lambda
specified_propvec <- 2 ^ (-thinlog2)
plot(empirical_propvec, specified_propvec,
     xlab = "Empirical Thinning Proportion",
     ylab = "Specified Thinning Proportion")
abline(0, 1, col = 2, lwd = 2)

}
\seealso{
\describe{
  \item{\code{\link{select_counts}}}{For subsampling the rows and columns
      of your real RNA-seq count matrix prior to applying binomial thinning.}
  \item{\code{\link{thin_diff}}}{For the more general thinning approach.}
  \item{\code{\link{thin_lib}}}{For thinning sample-wise instead of
      gene-wise.}
  \item{\code{\link{thin_all}}}{For thinning all counts uniformly.}
  \item{\code{\link{ThinDataToSummarizedExperiment}}}{For converting a
      ThinData object to a SummarizedExperiment object.}
  \item{\code{\link{ThinDataToDESeqDataSet}}}{For converting a
      ThinData object to a DESeqDataSet object.}
}
}
\author{
David Gerard
}
