% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/atos.R
\name{atos}
\alias{atos}
\title{adaptive three operator splitting (ATOS)}
\usage{
atos(
  X,
  y,
  type = "linear",
  prox_1,
  prox_2,
  pen_prox_1 = 0.5,
  pen_prox_2 = 0.5,
  max_iter = 5000,
  backtracking = 0.7,
  max_iter_backtracking = 100,
  tol = 1e-05,
  prox_1_opts = NULL,
  prox_2_opts = NULL,
  standardise = "l2",
  intercept = TRUE,
  x0 = NULL,
  u = NULL,
  verbose = FALSE
)
}
\arguments{
\item{X}{Input matrix of dimensions \eqn{p x n}. Can be a sparse matrix (using class \code{"sparseMatrix"} from the \code{Matrix} package)}

\item{y}{Output vector of dimension \eqn{n}. For \code{type="linear"} needs to be continuous and for \code{type="logistic"} needs to be a binary variable.}

\item{type}{The type of regression to perform. Supported values are: \code{"linear"} and \code{"logistic"}.}

\item{prox_1}{The proximal operator for the first function, \eqn{h(x)}.}

\item{prox_2}{The proximal operator for the second function, \eqn{g(x)}.}

\item{pen_prox_1}{The penalty for the first proximal operator. For the lasso, this would be the sparsity parameter, \eqn{\lambda}. If operator does not include a penalty, set to 1.}

\item{pen_prox_2}{The penalty for the second proximal operator.}

\item{max_iter}{Maximum number of ATOS iterations to perform.}

\item{backtracking}{The backtracking parameter, \eqn{\tau}, as defined in Pedregosa et. al. (2018).}

\item{max_iter_backtracking}{Maximum number of backtracking line search iterations to perform per global iteration.}

\item{tol}{Convergence tolerance for the stopping criteria.}

\item{prox_1_opts}{Optional argument for first proximal operator. For the group lasso, this would be the group IDs. Note: this must be inserted as a list.}

\item{prox_2_opts}{Optional argument for second proximal operator.}

\item{standardise}{Type of standardisation to perform on \code{X}:
\itemize{
\item \code{"l2"} standardises the input data to have \eqn{\ell_2} norms of one.
\item \code{"l1"} standardises the input data to have \eqn{\ell_1} norms of one.
\item \code{"sd"} standardises the input data to have standard deviation of one.
\item \code{"none"} no standardisation applied.
}}

\item{intercept}{Logical flag for whether to fit an intercept.}

\item{x0}{Optional initial vector for \eqn{x_0}.}

\item{u}{Optional initial vector for \eqn{u}.}

\item{verbose}{Logical flag for whether to print fitting information.}
}
\value{
An object of class \code{"atos"} containing:
\item{beta}{The fitted values from the regression. Taken to be the more stable fit between \code{x} and \code{u}, which is usually the former.}
\item{x}{The solution to the original problem (see Pedregosa et. al. (2018)).}
\item{u}{The solution to the dual problem (see Pedregosa et. al. (2018)).}
\item{z}{The updated values from applying the first proximal operator (see Pedregosa et. al. (2018)).}
\item{type}{Indicates which type of regression was performed.}
\item{success}{Logical flag indicating whether ATOS converged, according to \code{tol}.}
\item{num_it}{Number of iterations performed. If convergence is not reached, this will be \code{max_iter}.}
\item{certificate}{Final value of convergence criteria.}
\item{intercept}{Logical flag indicating whether an intercept was fit.}
}
\description{
Function for fitting adaptive three operator splitting (ATOS) with general convex penalties. Supports both linear and logistic regression, both with dense and sparse matrix implementations.
}
\details{
\code{atos()} solves convex minimization problems of the form
\deqn{
  f(x) + g(x) + h(x),
}
where \eqn{f} is convex and differentiable with \eqn{L_f}-Lipschitz gradient, and \eqn{g} and \eqn{h} are both convex.
The algorithm is not symmetrical, but usually the difference between variations are only small numerical values, which are filtered out.
However, both variations should be checked regardless, by looking at \code{x} and \code{u}. An example for the sparse-group lasso (SGL) is given.
}
\examples{
# specify a grouping structure
groups = c(rep(1:20, each=3),
          rep(21:40, each=4),
          rep(41:60, each=5),
          rep(61:80, each=6),
          rep(81:100, each=7))
# define proximal operators
L1_prox <- function(input, lambda){ # Lasso proximal operator
 out = sign(input) * pmax(0, abs(input) - lambda)
 return(out)
}
group_L1_prox = function(input,lambda,group_info){ 
 n_groups = length(unique(group_info))
 out = rep(0,length(input))
 for (i in 1:n_groups){
   grp_idx = which(group_info == unique(group_info)[i])
   if (lambda == 0 & norm(input[grp_idx],type="2") == 0){ # 0/0 = 0
     out[grp_idx] = 0
   } else {
     out[grp_idx] = max((1-(lambda/norm(input[grp_idx],type="2"))),0) * input[grp_idx]}
 }
 return(out)
}
# generate data
data = generate_toy_data(p=500, n=400, groups = groups, seed_id=3)
# run atos (the proximal functions can be found in utils.R)
out = atos(X=data$X, y=data$y, type="linear", prox_1 = L1_prox, prox_2 = group_L1_prox, 
standardise="none", intercept=FALSE, prox_2_opts = list(groups))
}
\references{
F. Pedregosa, G. Gidel (2018) \emph{Adaptive Three Operator Splitting}, \url{https://proceedings.mlr.press/v80/pedregosa18a.html}
}
