#' Collapse SHAP values
#'
#' Function used to collapse groups of columns in a SHAP matrix by rowwise summation.
#' A typical application is when the matrix of SHAP values is generated by a model with
#' one or multiple one-hot encoded variables and the explanations should be done using
#' the original variables.
#'
#' @param S A matrix of SHAP values.
#' @param collapse A named list of character vectors. Each vector specifies a group of
#' column names in \code{S} that should be collapsed to a single column by summation.
#' The name of the new column equals the name of the vector in \code{collapse}.
#' @param ... Currently unused.
#' @return A matrix with collapsed columns.
#' @export
#' @seealso \code{\link{shapviz}}.
#' @examples
#' S <- cbind(
#'   x = c(0.1, 0.1, 0.1),
#'   `age low` = c(0.2, -0.1, 0.1),
#'   `age mid` = c(0, 0.2, -0.2),
#'   `age high` = c(1, -1, 0)
#' )
#' collapse <- list(age = c("age low", "age mid", "age high"))
#' collapse_shap(S, collapse)
collapse_shap <- function(S, collapse = NULL, ...) {
  if (!is.matrix(S)) {
    stop("S must be a matrix.")
  }
  if (is.null(collapse) || length(collapse) == 0L || ncol(S) == 0L) {
    return(S)
  }
  stopifnot(
    "'S' must have column names" = !is.null(colnames(S)),
    "'collapse' must be a named list" = is.list(collapse) && !is.null(names(collapse)),
    "'collapse' can't have duplicated names" = !anyDuplicated(names(collapse))
  )
  u <- unlist(collapse, use.names = FALSE, recursive = FALSE)
  keep <- setdiff(colnames(S), u)
  stopifnot(
    "'collapse' cannot have overlapping vectors." = !anyDuplicated(u),
    "Values of 'collapse' should be in colnames(S)" = all(u %in% colnames(S)),
    "Names of 'collapse' must be different from untouched column names" =
      !any(names(collapse) %in% keep)
  )
  add <- do.call(
    cbind,
    lapply(collapse, function(z) rowSums(S[, z, drop = FALSE], na.rm = TRUE))
  )
  cbind(S[, keep, drop = FALSE], add)
}

