#' R6 Class implementing down-sampling in shiny app
#'
#' @export
#' @docType class
#' @format An \code{R6::R6Class} object
#' @importFrom R6 R6Class
#' @importFrom dplyr %>%
#' @description
#' This class includes original data, aggregation function(s) and plotly figure.
#' @examples
#'\donttest{
#' library(plotly)
#' data(noise_fluct)
#' p <- plot_ly(noise_fluct) %>%
#'   add_trace(x = ~sec, y = ~level, type = "scatter", mode = "lines")
#' d_app <- shiny_downsampler$new(p)
#' d_app$show_shiny()
#'}

# class decralation -------------------------------------------------------

shiny_downsampler <- R6::R6Class(
  "shiny_downsampler",
  inherit = abstract_downsampler,

# public members ----------------------------------------------------------

  public = list(
    #' @description
    #' Create a new downsampler
    #' @param figure,is_downsample,n_out,aggregator,legend_options,tz
    #' Arguments pass to the constructor of
    #' the \code{abstract_downsampler} class.
    initialize = function(
      figure = plotly::plot_ly(),
      is_downsample = TRUE,
      n_out = 1000L,
      aggregator = eLTTB_aggregator$new(),
      legend_options = list(
        downsample_prefix = '<b style="color:sandybrown">[R]</b> ',
        downsample_suffix = "",
        is_aggsize_shown = TRUE,
        agg_prefix = ' <i style="color:#fc9944">~',
        agg_suffix = "</i>"
      ),
      tz = Sys.timezone()
    ) {


      if (is.null(figure)) self$figure <- plotly::plot_ly()
      super$initialize(
        figure,
        is_downsample,
        n_out,
        aggregator,
        legend_options,
        tz
      )
    },



    #' @description
    #' update the trace data according to the relayout order.
    #' @param relayout_order Named list.
    #' The list generated by \code{plotlyjs_relayout},
    #' which is obtained using \code{plotly::event_data}.
    #' @param enforce Boolean.
    #' It it is \code{TRUE}, the figure will be updated even if
    #' \code{relayout_order} is \code{NULL}.
    cmpt_new_trace = function(relayout_order = list(), enforce = FALSE) {

      trace_update <- private$construct_update_data(relayout_order, enforce)

      if (is.null(trace_update) || nrow(trace_update) == 0) return()

      trace_idx_delete <- purrr::map(
        unique(trace_update$trace_uid),
        ~which(.x == purrr::map_chr(self$figure$x$data, ~.x$uid))
      ) %>%
        unlist()

      # delete traces if the stat_aggregator is changed to normal aggregator
      new_data <- trace_update %>%
        dplyr::select(trace_idx, trace_uid, aggregator, trace) %>%
        dplyr::mutate(
          trace = purrr::map2(
            trace_idx, trace,
            function(trace_idx, trace) {
              trace_exist <- self$figure$x$data[[trace_idx + 1]]

              trace_exist[c("text", "fill", "opacity", "hoveron")] <- NULL

              if("yupr" %in% names(trace) && "ylwr" %in% names(trace)) {
                trace_nom <- trace
                trace_nom[c("yupr", "ylwr")] <- NULL
                trace_rng <- private$cmpt_range_trace(trace)

                trace_nom_tmp <- trace_exist
                trace_rng_tmp <- trace_exist

                for (elem in names(trace_rng)) {
                  trace_rng_tmp[[elem]] <- trace_rng[[elem]]
                }
                for (elem in names(trace_nom)) {
                  trace_nom_tmp[[elem]] <- trace_nom[[elem]]
                }
                return(list(trace_rng_tmp, trace_nom_tmp))

              } else {

                trace_tmp <- trace_exist
                for (elem in names(trace)) {
                  trace_tmp[[elem]] <- trace[[elem]]
                }
                return(list(trace_tmp))

              }
            }
          )
        ) %>%
        tidyr::unnest(trace) %>%
        .$trace


      self$figure$x$data[trace_idx_delete] <- NULL
      self$figure$x$data <- new_data

      return(list(trace_idx_delete = trace_idx_delete, new_data = new_data))


    },

    #' @description
    #' Easy output of the shiny app.
    #' @param shiny_options Named list.
    #' Arguments passed to \code{shinyApp} as the options.
    #' @param width,height Character, optional.
    #' Arguments passed to \code{plotlyOutput}.
    #' By default, \code{100\%} and \code{600px}.
    show_shiny = function(
      shiny_options = list(), width = "100%", height = "600px"
    ) {
      # Define UI
      ui <- fluidPage(
        htmltools::div(
          selectizeInput(
            "agg_selector", label = "Aggregator",
            choices = list_aggregators() %>% str_subset("^[^(cus)]"),
            select = class(self$get_orig_data()[[1]]$aggregator)[1]
            ),
          numericInput(
            "n_out_input", label = "Number of samples",
            value = self$get_orig_data()[[1]]$n_out,
            step = 1, min = 1, max = 1e5
            ),
          style = "display:flex"
        ),
        plotlyOutput(outputId = "fig", width = width, height = height),
        "Relayout order:",
        verbatimTextOutput("relayout_order"),
        htmltools::br(),
        downloadButton("get_data", "Get shown data")
      )

      # Define server logic
      server <- function(input, output, session) {

        output$fig <- renderPlotly(self$figure)

        observe(
          {
            data       <- self$get_orig_data()
            uids       <- names(data)
            agg_classes<- purrr::map_chr(data, ~class(.x$aggregator)[1])
            n_outs     <- purrr::map_int(data, ~.x$n_out)

            agg_class_input <- input[["agg_selector"]]
            agg_input <- eval(parse(text = agg_class_input))$new()
            n_out_input <- input[["n_out_input"]]

            if (
              agg_classes[1] != agg_class_input || n_outs[1] != n_out_input
              ) {
              enforce_update <- TRUE
            } else {
              enforce_update <- FALSE
            }

            purrr::map(
              uids,
              ~self$set_orig_data(
                .x, aggregator = agg_input, n_out = n_out_input
                )
            )

            output[["relayout_order"]] <- renderPrint({
              tibble::as_tibble(plotly::event_data("plotly_relayout"))
              })

            updatePlotlyH(
              session, "fig", plotly::event_data("plotly_relayout"), self,
              enforce_update
              )
          },
          label = "figure_updater"
        )

        output[["get_data"]] <- downloadHandler(
          filename = function() {
            if (is.null(plotly::event_data("plotly_relayout"))) {
              rng <- "all"
            } else {
              rng <- paste(plotly::event_data("plotly_relayout"), collapse = "-")
            }
            paste0("data-", rng, ".csv")
          },
          content = function(file) {
            df_agg <- tibble(
                uid = names(self$get_orig_data()),
                agg = purrr::map_chr(
                  self$get_orig_data(),
                  ~class(.x$aggregator)[1]
                )
              )

            fig_data_to_tibble <- function(fig_data){
              agg <- df_agg$agg[df_agg$uid == fig_data$uid]
              if (str_detect(agg, "stat_aggregator$") &&
                  !is.null(fig_data$fill)) {
                n <- length(fig_data$x)
                tb <- tibble(
                  uid   = paste0(fig_data$uid, "rng"),
                  name  = fig_data$name,
                  x     = fig_data[["x"]][1:(n / 2)],
                  y_lwr = fig_data[["y"]][1:(n / 2)],
                  y_upr = fig_data[["y"]][n:(n / 2 + 1)]
                  )
              } else {
                tb <- tibble(
                  uid   = fig_data$uid,
                  name  = fig_data$name,
                  x     = fig_data[["x"]],
                  y     = fig_data[["y"]]
                )
              }
            }


            data <- purrr::map(
              self$get_figure_data(),
              ~fig_data_to_tibble(.x)
            ) %>%
              bind_rows() %>%
              tidyr::pivot_longer(c(y, y_upr, y_lwr), names_to = "key") %>%
              na.omit() %>%
              dplyr::mutate(uid = str_remove(uid, "rng$")) %>%
              tidyr::pivot_wider(names_from = "key", values_from = "value")

            write.csv(data, file)
          }
        )

      }

      # Return the application
      shinyApp(ui = ui, server = server, options = shiny_options)
    }

  ), # end of the public
  private = list(
  )
) # end of the class definition
