% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simStateSpace-sim-ssm-ou-i-vary.R
\name{SimSSMOUIVary}
\alias{SimSSMOUIVary}
\title{Simulate Data from an Ornstein–Uhlenbeck Model
using a State Space Model Parameterization
for n > 1 Individuals (Individual-Varying Parameters)}
\usage{
SimSSMOUIVary(
  n,
  mu0,
  sigma0,
  mu,
  phi,
  sigma,
  nu,
  lambda,
  theta,
  gamma_y = NULL,
  gamma_eta = NULL,
  x = NULL,
  type = 0,
  delta_t,
  time,
  burn_in = 0
)
}
\arguments{
\item{n}{Positive integer.
Number of individuals.}

\item{mu0}{Numeric vector.
Mean of initial latent variable values
(\eqn{\boldsymbol{\mu}_{\boldsymbol{\eta} \mid 0}}).}

\item{sigma0}{Numeric matrix.
The covariance matrix
of initial latent variable values
(\eqn{\boldsymbol{\Sigma}_{\boldsymbol{\eta} \mid 0}}).}

\item{mu}{List of numeric vectors.
Each element of the list
is the long-term mean or equilibrium level
(\eqn{\boldsymbol{\mu}}).}

\item{phi}{List of numeric matrices.
Each element of the list
is the rate of mean reversion,
determining how quickly the variable returns to its mean
(\eqn{\boldsymbol{\Phi}}).}

\item{sigma}{List of numeric matrices.
Each element of the list
is the matrix of volatility
or randomness in the process
(\eqn{\boldsymbol{\Sigma}}).}

\item{nu}{Numeric vector.
Vector of intercepts for the measurement model
(\eqn{\boldsymbol{\nu}}).}

\item{lambda}{Numeric matrix.
Factor loading matrix linking the latent variables
to the observed variables
(\eqn{\boldsymbol{\Lambda}}).}

\item{theta}{Numeric matrix.
The measurement error covariance matrix
(\eqn{\boldsymbol{\Theta}}).}

\item{gamma_y}{Numeric matrix.
Matrix relating the values of the covariate matrix
at time \code{t} to the observed variables at time \code{t}
(\eqn{\boldsymbol{\Gamma}_{\mathbf{y}}}).}

\item{gamma_eta}{Numeric matrix.
Matrix relating the values of the covariate matrix
at time \code{t} to the latent variables at time \code{t}
(\eqn{\boldsymbol{\Gamma}_{\boldsymbol{\eta}}}).}

\item{x}{Numeric matrix.
The matrix of observed covariates in \code{type = 1} or \code{type = 2}.
The number of rows should be equal to \code{time + burn_in}.}

\item{type}{Integer.
State space model type.
See Details for more information.}

\item{delta_t}{Numeric.
Time interval (\eqn{\delta_t}).}

\item{time}{Positive integer.
Number of time points to simulate.}

\item{burn_in}{Positive integer.
Number of burn-in points to exclude before returning the results.}
}
\value{
Returns an object of class \code{simstatespace}
which is a list with the following elements:
\itemize{
\item \code{call}: Function call.
\item \code{args}: Function arguments.
\item \code{data}: Generated data which is a list of length \code{n}.
Each element of \code{data} is a list with the following elements:
\itemize{
\item \code{id}: A vector of ID numbers of length \code{t}.
\item \code{time}: A vector time points of length \code{t}.
\item \code{y}: A \code{t} by \code{k} matrix of values for the manifest variables.
\item \code{eta}: A \code{t} by \code{p} matrix of values for the latent variables.
\item \code{x}: A \code{t} by \code{j} matrix of values for the covariates.
}
\item \code{fun}: Function used.
}
}
\description{
This function simulates data from an Ornstein–Uhlenbeck model
using a state space model parameterization
for \code{n > 1} individuals.
In this model,
the parameters can vary across individuals.
}
\details{
Parameters can vary across individuals
by providing a list of parameter values.
If the length of any of the parameters
(\code{mu0},
\code{sigma0},
\code{mu},
\code{phi},
\code{sigma},
\code{nu},
\code{lambda},
\code{theta},
\code{gamma_y}, or
\code{gamma_eta})
is less the \code{n},
the function will cycle through the available values.
}
\examples{
# prepare parameters
# In this example, phi varies across individuals
set.seed(42)
p <- k <- 2
iden <- diag(p)
n <- 5
mu0 <- list(c(-3.0, 1.5))
sigma0 <- list(iden)
mu <- list(c(5.76, 5.18))
phi <- list(
  as.matrix(Matrix::expm(diag(x = -0.1, nrow = k))),
  as.matrix(Matrix::expm(diag(x = -0.2, nrow = k))),
  as.matrix(Matrix::expm(diag(x = -0.3, nrow = k))),
  as.matrix(Matrix::expm(diag(x = -0.4, nrow = k))),
  as.matrix(Matrix::expm(diag(x = -0.5, nrow = k)))
)
sigma <- list(
  matrix(data = c(2.79, 0.06, 0.06, 3.27), nrow = p)
)
nu <- list(rep(x = 0, times = k))
lambda <- list(diag(k))
theta <- list(diag(x = 0.50, nrow = k))
delta_t <- 0.10
time <- 50
burn_in <- 0
gamma_y <- gamma_eta <- list(0.10 * diag(k))
x <- lapply(
  X = seq_len(n),
  FUN = function(i) {
    return(
      matrix(
        data = rnorm(n = k * (time + burn_in)),
        ncol = k
      )
    )
  }
)

# Type 0
ssm <- SimSSMOUIVary(
  n = n,
  mu0 = mu0,
  sigma0 = sigma0,
  mu = mu,
  phi = phi,
  sigma = sigma,
  nu = nu,
  lambda = lambda,
  theta = theta,
  type = 0,
  delta_t = delta_t,
  time = time,
  burn_in = burn_in
)

plot(ssm)

# Type 1
ssm <- SimSSMOUIVary(
  n = n,
  mu0 = mu0,
  sigma0 = sigma0,
  mu = mu,
  phi = phi,
  sigma = sigma,
  nu = nu,
  lambda = lambda,
  theta = theta,
  gamma_eta = gamma_eta,
  x = x,
  type = 1,
  delta_t = delta_t,
  time = time,
  burn_in = burn_in
)

plot(ssm)

# Type 2
ssm <- SimSSMOUIVary(
  n = n,
  mu0 = mu0,
  sigma0 = sigma0,
  mu = mu,
  phi = phi,
  sigma = sigma,
  nu = nu,
  lambda = lambda,
  theta = theta,
  gamma_y = gamma_y,
  gamma_eta = gamma_eta,
  x = x,
  type = 2,
  delta_t = delta_t,
  time = time,
  burn_in = burn_in
)

plot(ssm)

}
\references{
Chow, S.-M., Losardo, D., Park, J., & Molenaar, P. C. M. (2023).
Continuous-time dynamic models: Connections to structural equation models
and other discrete-time models.
In R. H. Hoyle (Ed.), Handbook of structural equation modeling (2nd ed.).
The Guilford Press.

Uhlenbeck, G. E., & Ornstein, L. S. (1930).
On the theory of the brownian motion.
\emph{Physical Review}, \emph{36}(5), 823–841.
\doi{10.1103/physrev.36.823}
}
\seealso{
Other Simulation of State Space Models Data Functions: 
\code{\link{OU2SSM}()},
\code{\link{SimSSMFixed}()},
\code{\link{SimSSMIVary}()},
\code{\link{SimSSMLinGrowthIVary}()},
\code{\link{SimSSMLinGrowth}()},
\code{\link{SimSSMOUFixed}()},
\code{\link{SimSSMOU}()},
\code{\link{SimSSMVARFixed}()},
\code{\link{SimSSMVARIVary}()},
\code{\link{SimSSMVAR}()},
\code{\link{SimSSM}()}
}
\author{
Ivan Jacob Agaloos Pesigan
}
\concept{Simulation of State Space Models Data Functions}
\keyword{ou}
\keyword{sim}
\keyword{simStateSpace}
