% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/compilation.R
\name{slim}
\alias{slim}
\title{Run a slendr model in SLiM}
\usage{
slim(
  model,
  sequence_length,
  recombination_rate,
  samples = NULL,
  output = NULL,
  burnin = 0,
  max_attempts = 1,
  spatial = !is.null(model$world),
  coalescent_only = TRUE,
  method = c("batch", "gui"),
  random_seed = NULL,
  verbose = FALSE,
  load = TRUE,
  locations = NULL,
  slim_path = NULL,
  sampling = NULL
)
}
\arguments{
\item{model}{Model object created by the \code{compile} function}

\item{sequence_length}{Total length of the simulated sequence (in base-pairs)}

\item{recombination_rate}{Recombination rate of the simulated sequence (in
recombinations per basepair per generation)}

\item{samples}{A data frame of times at which a given number of individuals
should be remembered in the tree-sequence (see \code{schedule_sampling} for a
function that can generate the sampling schedule in the correct format). If
missing, only individuals present at the end of the simulation will be
recorded in the tree-sequence output file.}

\item{output}{Path to the output tree-sequence file. If \code{NULL} (the default),
tree sequence will be saved to a temporary file.}

\item{burnin}{Length of the burnin (in model's time units, i.e. years)}

\item{max_attempts}{How many attempts should be made to place an offspring
near one of its parents? Serves to prevent infinite loops on the SLiM
backend. Default value is 1.}

\item{spatial}{Should the model be executed in spatial mode? By default, if a
world map was specified during model definition, simulation will proceed in
a spatial mode.}

\item{coalescent_only}{Should \code{initializeTreeSeq(retainCoalescentOnly =
  <...>)} be set to \code{TRUE} (the default) or \code{FALSE}? See
"retainCoalescentOnly" in the SLiM manual for more detail.}

\item{method}{How to run the script? ("gui" - open in SLiMgui, "batch" - run
on the command-line)}

\item{random_seed}{Random seed (if missing, SLiM's own seed will be used)}

\item{verbose}{Write the SLiM output log to the console (default
\code{FALSE})?}

\item{load}{Should the final tree sequence be immediately loaded and returned?
Default is \code{TRUE}. The alternative (\code{FALSE}) is useful when a tree-sequence
file is written to a custom location to be loaded at a later point.}

\item{locations}{If \code{NULL}, locations are not saved. Otherwise, the
path to the file where locations of each individual throughout the simulation
will be saved (most likely for use with \code{animate_model}).}

\item{slim_path}{Optional way to specify path to an appropriate SLiM binary (this is useful
if the \code{slim} binary is not on the \code{$PATH}).}

\item{sampling}{Deprecated in favor of \code{samples}.}
}
\value{
A tree-sequence object loaded via Python-R reticulate interface function \code{ts_load}
(internally represented by the Python object \code{tskit.trees.TreeSequence})
}
\description{
This function will execute a SLiM script generated by the \code{compile}
function during the compilation of a slendr demographic model.
}
\examples{
\dontshow{check_dependencies(python = TRUE, slim = TRUE) # make sure dependencies are present
}
# load an example model
model <- read_model(path = system.file("extdata/models/introgression", package = "slendr"))

# afr and eur objects would normally be created before slendr model compilation,
# but here we take them out of the model object already compiled for this
# example (in a standard slendr simulation pipeline, this wouldn't be necessary)
afr <- model$populations[["AFR"]]
eur <- model$populations[["EUR"]]
chimp <- model$populations[["CH"]]

# schedule the sampling of a couple of ancient and present-day individuals
# given model at 20 ky, 10 ky, 5ky ago and at present-day (time 0)
modern_samples <- schedule_sampling(model, times = 0, list(afr, 5), list(eur, 5), list(chimp, 1))
ancient_samples <- schedule_sampling(model, times = c(30000, 20000, 10000), list(eur, 1))

# sampling schedules are just data frames and can be merged easily
samples <- rbind(modern_samples, ancient_samples)

# run a simulation using the SLiM back end from a compiled slendr model object and return
# a tree-sequence output
ts <- slim(model, sequence_length = 1e5, recombination_rate = 0, samples = samples)

# automatic loading of a simulated output can be prevented by `load = FALSE`, which can be
# useful when a custom path to a tree-sequence output is given for later downstream analyses
output_file <- tempfile(fileext = ".trees")
slim(model, sequence_length = 1e5, recombination_rate = 0, samples = samples,
     output = output_file, load = FALSE)
# ... at a later stage:
ts <- ts_load(output_file, model)

ts
}
