% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SDA_query.R
\name{SDA_query}
\alias{SDA_query}
\title{Query Soil Data Access}
\usage{
SDA_query(q, dsn = NULL)
}
\arguments{
\item{q}{character. A valid T-SQL query surrounded by double quotes.}

\item{dsn}{character. Default: \code{NULL} uses Soil Data Access remote data source via REST API. Alternately, \code{dsn} may be a file path to an SQLite database using the SSURGO schema, or a \code{DBIConnection} that has already been created.}
}
\value{
A data.frame result for queries that return a single table. A list of data.frame for queries that return multiple tables. \code{NULL} if result is empty, and \code{try-error} on error.
}
\description{
Submit a query to the Soil Data Access (SDA) REST/JSON web-service and return the results as a data.frame. There is a 100,000 record and 32Mb JSON serialization limit per query. Queries should contain a WHERE clause or JOIN condition to limit the number of rows affected / returned. Consider wrapping calls to \code{SDA_query()} in a function that can iterate over logical chunks (e.g. areasymbol, mukey, cokey, etc.). The function \code{makeChunks()} can help with such iteration. All usages of \code{SDA_query()} should handle the possibility of a \code{try-error} result in case the web service connection is down or if an invalid query is passed to the endpoint.
}
\details{
The SDA website can be found at \url{https://sdmdataaccess.nrcs.usda.gov} and query examples can be found at \url{https://sdmdataaccess.nrcs.usda.gov/QueryHelp.aspx}. A library of query examples can be found at \url{https://nasis.sc.egov.usda.gov/NasisReportsWebSite/limsreport.aspx?report_name=SDA-SQL_Library_Home}.

SSURGO (detailed soil survey) and STATSGO (generalized soil survey) data are stored together within SDA. This means that queries that don't specify an area symbol may result in a mixture of SSURGO and STATSGO records. See the examples below and the \href{http://ncss-tech.github.io/AQP/soilDB/SDA-tutorial.html}{SDA Tutorial} for details.
}
\note{
This function requires the \code{httr}, \code{jsonlite}, and \code{xml2} packages
}
\examples{
\dontshow{if (curl::has_internet() && requireNamespace("wk", quietly = TRUE)) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
\dontshow{\}) # examplesIf}
\donttest{
  ## get SSURGO export date for all soil survey areas in California
  # there is no need to filter STATSGO
  # because we are filtering on SSURGO area symbols
  q <- "SELECT areasymbol, saverest FROM sacatalog WHERE areasymbol LIKE 'CA\%';"
  x <- SDA_query(q)
  head(x)


  ## get SSURGO component data associated with the
  ## Amador series / major component only
  # this query must explicitly filter out STATSGO data
  q <- "SELECT cokey, compname, comppct_r FROM legend
    INNER JOIN mapunit mu ON mu.lkey = legend.lkey
    INNER JOIN component co ON mu.mukey = co.mukey
    WHERE legend.areasymbol != 'US' AND compname = 'Amador';"

  res <- SDA_query(q)
  str(res)

  ## get component-level data for a specific soil survey area (Yolo county, CA)
  # there is no need to filter STATSGO because the query contains
  # an implicit selection of SSURGO data by areasymbol
  q <- "SELECT
    component.mukey, cokey, comppct_r, compname, taxclname,
    taxorder, taxsuborder, taxgrtgroup, taxsubgrp
    FROM legend
    INNER JOIN mapunit ON mapunit.lkey = legend.lkey
    LEFT OUTER JOIN component ON component.mukey = mapunit.mukey
    WHERE legend.areasymbol = 'CA113' ;"

  res <- SDA_query(q)
  str(res)

  ## get tabular data based on result from spatial query
  # there is no need to filter STATSGO because
  # SDA_Get_Mukey_from_intersection_with_WktWgs84() implies SSURGO
  p <- wk::as_wkt(wk::rct(-120.9, 37.7, -120.8, 37.8))
  q <- paste0("SELECT mukey, cokey, compname, comppct_r FROM component
      WHERE mukey IN (SELECT DISTINCT mukey FROM
      SDA_Get_Mukey_from_intersection_with_WktWgs84('", p,
       "')) ORDER BY mukey, cokey, comppct_r DESC")

   x <- SDA_query(q)
   str(x)
}
}
\seealso{
\code{\link[=SDA_spatialQuery]{SDA_spatialQuery()}}
}
\author{
D.E. Beaudette, A.G Brown
}
\keyword{manip}
