% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/group_lines.R
\name{group_lines}
\alias{group_lines}
\title{Group Lines}
\usage{
group_lines(
  DT = NULL,
  threshold = NULL,
  projection = NULL,
  id = NULL,
  coords = NULL,
  timegroup = NULL,
  sortBy = NULL,
  splitBy = NULL,
  spLines = NULL
)
}
\arguments{
\item{DT}{input data.table}

\item{threshold}{The width of the buffer around the lines in the units of the
projection. Supply 0 to compare intersection without buffering.}

\item{projection}{character string defining the EPSG code. For example, for UTM zone 21N (EPSG 32736), the projection argument is "+init=epsg:32736". See details.}

\item{id}{Character string of ID column name}

\item{coords}{Character vector of X coordinate and Y coordinate column names}

\item{timegroup}{timegroup field in the DT upon which the grouping will be
calculated}

\item{sortBy}{Character string of date time column(s) to sort rows by. Must
be a POSIXct.}

\item{splitBy}{(optional) character string or vector of grouping column
name(s) upon which the grouping will be calculated}

\item{spLines}{Alternatively to providing a DT, provide a SpatialLines object
created with the sp package. If a spLines object is provided, groups cannot
be calculated by a    timegroup or splitBy.}
}
\value{
\code{group_lines} returns the input \code{DT} appended with a
\code{group} column.

This column represents the spatial (and if \code{timegroup} was provided -
spatiotemporal) group calculated by overlapping lines. As with the other
grouping functions,  the actual value of group is arbitrary and represents
the identity of a given group where 1 or more individuals are assigned to a
group. If the data was reordered, the group may change, but the contents of
each group would not.

A message is returned when a column named \code{group} already exists in
the input \code{DT}, because it will be overwritten.
}
\description{
\code{group_lines} groups rows into spatial groups by creating trajectories
and grouping based on spatial overlap. The function accepts a
\code{data.table} with relocation data, individual identifiers and a
\code{threshold}. The relocation data is transformed into \code{SpatialLines}
and overlapping \code{SpatialLines} are grouped. The \code{threshold}
argument is used to specify the criteria for distance between lines.
Relocation data should be in two columns representing the X and Y
coordinates.
}
\details{
The \code{DT} must be a \code{data.table}. If your data is a
\code{data.frame}, you can convert it by reference using
\code{\link[data.table:setDT]{data.table::setDT}}.

The \code{id}, \code{coords}, \code{sortBy} (and optional \code{timegroup}
and \code{splitBy}) arguments expect the names of respective columns in
\code{DT} which correspond to the individual identifier, X and Y coordinates,
sorting, timegroup (generated by \code{group_times}) and additional grouping
columns.

The \code{projection} argument expects a character string defining the EPSG code. For example, for UTM zone 21N (EPSG 32736), the projection argument is "+init=epsg:32736". See \url{https://spatialreference.org} for a list of EPSG codes. Please note, R spatial has followed updates to GDAL and PROJ for handling projections, see more at \url{https://www.r-spatial.org/r/2020/03/17/wkt.html}.

The \code{sortBy} is used to order the input \code{data.table} when creating
\code{SpatialLines}. It must a \code{POSIXct} to ensure the rows are sorted
by date time.

The \code{threshold} must be provided in the units of the coordinates. The
\code{threshold} can be equal to 0 if strict overlap is required, else it
needs to be greater than 0. The coordinates must be planar coordinates (e.g.:
UTM). In the case of UTM, a \code{threshold} = 50 would indicate a 50m
distance threshold.

The \code{timegroup} argument is optional, but recommended to pair with
\code{\link{group_times}}. The intended framework is to group rows temporally
with \code{\link{group_times}} then spatially with \code{group_lines} (or
\code{\link{group_pts}}, \code{\link{group_polys}}). With \code{group_lines},
pick a relevant \code{group_times} \code{threshold} such as \code{'1 day'} or
\code{'7 days'} which is informed by your study species and system.

The \code{splitBy} argument offers further control over grouping. If within
your \code{DT}, you have multiple populations, subgroups or other distinct
parts, you can provide the name of the column which identifies them to
\code{splitBy}. The grouping performed by \code{group_lines} will only
consider rows within each \code{splitBy} subgroup.
}
\examples{
# Load data.table
library(data.table)

# Read example data
DT <- fread(system.file("extdata", "DT.csv", package = "spatsoc"))

# Subset only individuals A, B, and C
DT <- DT[ID \%in\% c('A', 'B', 'C')]

# Cast the character column to POSIXct
DT[, datetime := as.POSIXct(datetime, tz = 'UTC')]

# EPSG code for example data
utm <- '+init=epsg:32736'

\donttest{group_lines(DT, threshold = 50, projection = utm, sortBy = 'datetime',
            id = 'ID', coords = c('X', 'Y'))}

## Daily movement tracks
# Temporal grouping
group_times(DT, datetime = 'datetime', threshold = '1 day')

# Subset only first 50 days
DT <- DT[timegroup < 25]

# Spatial grouping
group_lines(DT, threshold = 50, projection = utm,
            id = 'ID', coords = c('X', 'Y'),
            timegroup = 'timegroup', sortBy = 'datetime')

## Daily movement tracks by population
group_lines(DT, threshold = 50, projection = utm,
            id = 'ID', coords = c('X', 'Y'),
            timegroup = 'timegroup', sortBy = 'datetime',
            splitBy = 'population')
}
\seealso{
\code{\link{build_lines}} \code{\link{group_times}}

Other Spatial grouping: 
\code{\link{group_polys}()},
\code{\link{group_pts}()}
}
\concept{Spatial grouping}
