% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{Gauss_Legen}
\alias{Gauss_Legen}
\alias{Gauss_Legen_nodes}
\alias{Gauss_Legen_weights}
\title{Gauss--Legendre quadrature}
\usage{
Gauss_Legen_nodes(a = -1, b = 1, N = 40L)

Gauss_Legen_weights(a = -1, b = 1, N = 40L)
}
\arguments{
\item{a, b}{scalars giving the interval \eqn{(a, b)}. Defaults to
\eqn{(-1, 1)}.}

\item{N}{number of points used in the Gauss--Legendre quadrature. The
following choices are supported: \code{5}, \code{10}, \code{20},
\code{40}, \code{80}, \code{160}, \code{320}, \code{640}, \code{1280},
\code{2560}, and \code{5120}. Defaults to \code{40}.}
}
\value{
A matrix of size \code{c(N, 1)} with the nodes \eqn{x_k}
(\code{Gauss_Legen_nodes}) or the corresponding weights \eqn{w_k}
(\code{Gauss_Legen_weights}).
}
\description{
Convenience for computing the nodes \eqn{x_k} and weights
\eqn{w_k} of the \emph{Gauss--Legendre} quadrature formula
in \eqn{(a, b)}:
\deqn{\int_a^b f(x) w(x)\,\mathrm{d}x\approx\sum_{k=1}^N w_k f(x_k).}{
\int_a^b f(x) dx\approx\sum_{k=1}^N w_k f(x_k)}.
}
\details{
For \eqn{C^\infty} functions, Gauss--Legendre quadrature
can be very efficient. It is exact for polynomials up to degree
\eqn{2N - 1}.

The nodes and weights up to \eqn{N = 80} were retrieved from
\href{https://dlmf.nist.gov/3.5#v}{NIST} and have \eqn{10^{-21}} precision.
For \eqn{N = 160} onwards, the nodes and weights were computed with the
\code{gauss.quad} function from the \href{https://CRAN.R-project.org/package=statmod}{
\code{statmod}} package (Smyth, 1998), and have \eqn{10^{-15}} precision.
}
\examples{
## Integration of a smooth function in (1, 10)

# Weights and nodes for integrating
x_k <- Gauss_Legen_nodes(a = 1, b = 10, N = 40)
w_k <- Gauss_Legen_weights(a = 1, b = 10, N = 40)

# Check quadrature
f <- function(x) sin(x) * x^2 - log(x + 1)
integrate(f, lower = 1, upper = 10, rel.tol = 1e-12)
sum(w_k * f(x_k))

# Exact for polynomials up to degree 2 * N - 1
f <- function(x) (((x + 0.5) / 1e3)^5 - ((x - 0.5)/ 5)^4 +
  ((x - 0.25) / 10)^2 + 1)^20
sum(w_k * f(x_k))
integrate(f, lower = -1, upper = 1, rel.tol = 1e-12)

## Integration on (0, pi)

# Weights and nodes for integrating
th_k <- Gauss_Legen_nodes(a = 0, b = pi, N = 40)
w_k <- Gauss_Legen_weights(a = 0, b = pi, N = 40)

# Check quadrature
p <- 4
psi <- function(th) -sin(th / 2)
w <- function(th) sin(th)^(p - 2)
integrate(function(th) psi(th) * w(th), lower = 0, upper = pi,
          rel.tol = 1e-12)
sum(w_k * psi(th_k) * w(th_k))

# Integral with Gegenbauer polynomial
k <- 3
C_k <- function(th) drop(Gegen_polyn(theta = th, k = k, p = p))
integrate(function(th) psi(th) * C_k(th) * w(th), lower = 0, upper = pi,
          rel.tol = 1e-12)
th_k <- drop(Gauss_Legen_nodes(a = 0, b = pi, N = 80))
w_k <- drop(Gauss_Legen_weights(a = 0, b = pi, N = 80))
sum(w_k * psi(th_k) * C_k(th_k) * w(th_k))
}
\references{
\emph{NIST Digital Library of Mathematical Functions}. Release
1.0.20 of 2018-09-15. F. W. J. Olver, A. B. Olde Daalhuis, D. W. Lozier,
B. I. Schneider, R. F. Boisvert, C. W. Clark, B. R. Miller,
and B. V. Saunders, eds. \url{https://dlmf.nist.gov/}

Smyth, G. K. (1998). Numerical integration. In: \emph{Encyclopedia of
Biostatistics}, P. Armitage and T. Colton (eds.), Wiley, London,
pp. 3088-3095.
}
