% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/scan.test.R
\name{scan.test}
\alias{scan.test}
\title{Compute the scan test.}
\usage{
scan.test(formula = NULL, data = NULL, fx = NULL, coor = NULL,
case = NULL, nv = NULL, nsim = NULL, distr = NULL, windows = "circular",
alternative = "High", minsize = 1, control = list())
}
\arguments{
\item{formula}{a symbolic description of the factor (optional).}

\item{data}{an (optional) data frame or a sf object containing the variable to testing for.}

\item{fx}{a factor (optional).}

\item{coor}{(optional) coordinates of observations.}

\item{case}{Only for Bernoulli distribution. A element of factor, there are cases and non-cases for testing for cases versus non-cases}

\item{nv}{Maximum windows size, default nv = N/2. The algorithm scan for clusters of geographic size between 1
and the upper limit (nv) defined by the user.}

\item{nsim}{Number of permutations.}

\item{distr}{distribution of the spatial process: "bernoulli" for two levels or "multinomial" for three or more levels.}

\item{windows}{a string to select the type of cluster "circular" (default) of "elliptic".}

\item{alternative}{Only for Bernoulli spatial process. A character string specifying the type of cluster, must be one
of "High" (default), "Both" or "Low".}

\item{minsize}{Minimum number of observations inside of Most Likely Cluster and secondary clusters.}

\item{control}{List of additional control arguments.}
}
\value{
A object of the \emph{htest} and \emph{scantest} class
  \tabular{ll}{
    \code{method} \tab The type of test applied ().\cr
    \code{fx} \tab Factor included as input to get the scan test.\cr
    \code{MLC} \tab Observations included into the Most Likelihood Cluster (MLC).\cr
    \code{statistic} \tab Value of the scan test (maximum Log-likelihood ratio). \cr
    \code{N} \tab Total number of observations.\cr
    \code{nn} \tab Windows used to get the cluster.\cr
    \code{nv} \tab Maximum number of observations into the cluster.\cr
    \code{data.name} \tab A character string giving the name of the factor.\cr
    \code{coor} \tab coordinates.\cr
    \code{alternative} \tab Only for bernoulli spatial process. A character string describing
    the alternative hypothesis select by the user.\cr
    \code{p.value} \tab p-value of the scan test.\cr
    \code{cases.expect} \tab Expected cases into the MLC.\cr
    \code{cases.observ} \tab Observed cases into the MLC.\cr
    \code{nsim} \tab Number of permutations.\cr
    \code{scan.mc} \tab a (nsim x 1) vector with the loglik values under bootstrap permutation.\cr
    \code{secondary.clusters} \tab a list with the observations included into the secondary clusters.\cr
    \code{loglik.second} \tab a vector with the value of the secondary scan tests (maximum Log-likelihood ratio).\cr
    \code{p.value.secondary} \tab a vector with the p-value of the secondary scan tests.\cr
    \code{Alternative.MLC} \tab A vector with the observations included in another cluster with the same loglik than MLC.\cr
    }
}
\description{
This function compute the scan test for Bernoulli and Multinomial categorical spatial process.
}
\details{
Two alternative sets of arguments can be included in this function to compute the scan test:

  \tabular{ll}{
    \code{Option 1} \tab A factor (fx) and coordinates (coor). \cr
    \code{Option 2} \tab A sf object (data) and the formula to specify the factor.
    The function consider the coordinates of the centroids of the elements of th sf object. \cr
    }

 The spatial scan statistics are widely used in epidemiology, criminology or ecology.
 Their purpose is to analyze the spatial distribution of points or geographical
 regions by testing the hypothesis of spatial randomness distribution on the
 basis of different distributions (e.g. Bernoulli, Poisson or Normal distributions).
 The \code{scan.test} function obtain the scan statistic for the Bernoulli and Multinomial distribution.

 To test independence in a spatial process, under the null, the type of windows is irrelevant but under the alternative the elliptic
 windows can to identify with more precision the cluster.

 For big data sets (N >>) the windows = "elliptic" can be so slowly
}
\section{Control arguments}{

  \tabular{ll}{
    \code{seedinit} \tab Numerical value for the seed (only for boot version). Default value seedinit=123 \cr
      }
}

\examples{

# Case 1: Scan test bernoulli
data(provinces_spain)
sf::sf_use_s2(FALSE)
provinces_spain$Male2Female <- factor(provinces_spain$Male2Female > 100)
levels(provinces_spain$Male2Female) = c("men","woman")
formula <- ~ Male2Female
scan <- scan.test(formula = formula, data = provinces_spain, case="men",
nsim = 99, distr = "bernoulli")
print(scan)
summary(scan)
plot(scan, sf = provinces_spain)

## With maximum number of neighborhood
scan <- scan.test(formula = formula, data = provinces_spain, case = "woman",
nsim = 99, distr = "bernoulli")
print(scan)
plot(scan, sf = provinces_spain)

\donttest{
## With elliptic windows
scan <- scan.test(formula = formula, data = provinces_spain, case = "men", nv = 25,
nsim = 99, distr = "bernoulli", windows ="elliptic")
print(scan)
scan <- scan.test(formula = formula, data = provinces_spain, case = "men", nv = 15,
nsim = 99, distr = "bernoulli", windows ="elliptic", alternative = "Low")
print(scan)
plot(scan, sf = provinces_spain)

# Case 2: scan test multinomial
data(provinces_spain)
provinces_spain$Older <- cut(provinces_spain$Older, breaks = c(-Inf,19,22.5,Inf))
levels(provinces_spain$Older) = c("low","middle","high")
formula <- ~ Older
scan <- scan.test(formula = formula, data = provinces_spain, nsim = 99, distr = "multinomial")
print(scan)
plot(scan, sf = provinces_spain)

# Case 3: scan test multinomial
data(FastFood.sf)
sf::sf_use_s2(FALSE)
formula <- ~ Type
scan <- scan.test(formula = formula, data = FastFood.sf, nsim = 99,
distr = "multinomial", windows="elliptic", nv = 50)
print(scan)
summary(scan)
plot(scan, sf = FastFood.sf)

# Case 4: DGP two categories
N <- 1500
cx <- runif(N)
cy <- runif(N)
listw <- spdep::knearneigh(cbind(cx,cy), k = 10)
p <- c(1/2,1/2)
rho <- 0.5
fx <- dgp.spq(p = p, listw = listw, rho = rho)
scan <- scan.test(fx = fx, nsim = 99, case = "A", nv = 50, coor = cbind(cx,cy),
distr = "bernoulli",windows="circular")
print(scan)
plot(scan)

# Case 5: DGP three categories
N <- 200
cx <- runif(N)
cy <- runif(N)
listw <- spdep::knearneigh(cbind(cx,cy), k = 10)
p <- c(1/3,1/3,1/3)
rho <- 0.5
fx <- dgp.spq(p = p, listw = listw, rho = rho)
scan <- scan.test(fx = fx, nsim = 19, coor = cbind(cx,cy), nv = 30,
distr = "multinomial", windows = "elliptic")
print(scan)
plot(scan)
}
}
\references{
\itemize{
    \item Kulldorff M, Nagarwalla N. (1995).
    Spatial disease clusters: Detection and Inference.
      \emph{Statistics in Medicine}. 14:799-810
    \item Jung I, Kulldorff M, Richard OJ (2010).
    A spatial scan statistic for multinomial data.
      \emph{Statistics in Medicine}. 29(18), 1910-1918
     \item Páez, A., López-Hernández, F.A., Ortega-García, J.A., Ruiz, M. (2016).
     Clustering and co-occurrence of cancer types: A comparison of techniques with
     an application to pediatric cancer in Murcia, Spain.
     \emph{Spatial Analysis in Health Geography}, 69-90.

  }
}
\seealso{
\code{\link{local.sp.runs.test}}, \code{\link{dgp.spq}}, \code{\link{Q.test}},
}
\author{
\tabular{ll}{
  Fernando López  \tab \email{fernando.lopez@upct.es} \cr
  Román Mínguez  \tab \email{roman.minguez@uclm.es} \cr
  Antonio Páez \tab \email{paezha@gmail.com} \cr
  Manuel Ruiz \tab \email{manuel.ruiz@upct.es} \cr
  }
}
