% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/str2str_functions.R
\name{cbind<-}
\alias{cbind<-}
\title{Add Columns to Data Objects}
\usage{
cbind(data, after = ncol(data), col.nm = NULL, overwrite = TRUE) <- value
}
\arguments{
\item{data}{data.frame or matrix of data.}

\item{after}{either an integer vector with length 1 or a character vector of
length 1 specifying where to add \code{value}. If an integer vector, it is the
position of a column. If a character vector it is the column with that name.
Similar to \code{append}, use 0L if you want the added columns to be first.}

\item{col.nm}{character vector of length equal to \code{NCOL(value)} that
specifies the colnames of \code{value} once added to \code{data} as columns.
This is an optional argument that defaults to NULL where the pre-existing colnames
of \code{value} are used.}

\item{overwrite}{logical vector of length 1 specifying whether columns from
\code{value} or \code{col.nm} should overwrite columns in \code{data} with the
same colnames. Note, if \code{overwrite} = FALSE, repeat colnames are possible
similar to \code{cbind}.}

\item{value}{data.frame, matrix, or atomic vector to be added as columns to
\code{data}. If a data.frame or matrix, it must have the same nrow as \code{data}.
If an atomic vector, it must have length equal to nrow of \code{data}.}
}
\value{
Like other similar functions (e.g., \code{`names<-`} and \code{`[<-`}),
\code{`cbind<-`} does not appear to have a return object. However, it technically
does as a side effect. The argument \code{data} will have been changed such that
\code{value} has been added as columns. If a traditional return object is desired,
and no side effects, then it can be called like a traditional function:
dat2 <- `cbind<-`(dat1, value = add1).
}
\description{
\code{`cbind<-`} adds columns to data objects as a side effect. The purpose of
the function is to replace the need to use dat2 <- cbind(dat1, add1);
dat3 <- cbind(dat2, add2); dat4 <- cbind(dat3, add3), etc. For data.frames,
it functions similarly to \code{`[<-.data.frame`}, but allows you to specify the
location of the columns similar to \code{append} (vs. \code{c}) and overwrite
columns with the same colnames. For matrices, it offers more novel functionality
since \code{`[<-.matrix`} does not exist.
}
\details{
Some traditional R folks may find this function uncomfortable. R is famous for limiting
side effects, except for a few notable exceptions (e.g., \code{`[<-`} and \code{`names<-`}).
Part of the reason is that side effects can be computationally inefficient in R.
The entire object often has to be re-constructed and re-saved to memory. For
example, a more computationally efficient alternative to cbind(dat) <- add1;
cbind(dat) <- add2; cbind(dat) <- add3 is dat1 <- do.call(what = cbind,
args = list(dat, add1, add2, add3)). However, \code{`cbind<-`} was not created
for R programming use when computational efficiency is valued; it is created
for R interactive use when user convenience is valued.

Similar to \code{`cbind`}, \code{`cbind<-`} works with both data.frames and matrices.
This is because \code{`cbind`} is a generic function with a default method that
works with matrices and a data.frame method that works with data.frames. Similar
to \code{`cbind`}, if colnames of \code{value} are not given and \code{col.nm}
is left NULL, then the colnames of the return object are automatically created
and can be dissatisfying.
}
\examples{
attitude2 <- attitude
cbind(attitude2) <- rowMeans(attitude2) # defaults to colnames = "value"
attitude2["value"] <- NULL
cbind(attitude2, col.nm = "mean") <- rowMeans(attitude2) # colnames specified by `col.nm`
attitude2["mean"] <- NULL
cbind(attitude2, after = "privileges", col.nm = c("mean","sum")) <-
   cbind(rowMeans(attitude2), rowSums(attitude2)) # `value` can be a matrix
attitude2[c("mean","sum")] <- NULL
attitude2 <- `cbind<-`(data = attitude2, value = rowMeans(attitude2)) # traditional call
attitude2["value"] <- NULL
cbind(attitude2, after = "privileges", col.nm = "mean") <-
   rowMeans(attitude2) # `data` can be a matrix
cbind(attitude2) <- data.frame("mean" = rep.int(x = "mean", times = 30L)) # overwrite = TRUE
cbind(attitude2, overwrite = FALSE) <-
   data.frame("mean" = rep.int(x = "mean", times = 30L)) # overwrite = FALSE
cbind(attitude2) <- data.frame("mean" = rep.int(x = "MEAN", times = 30L),
   "sum" = rep.int(x = "SUM", times = 30L)) # will overwrite only the first "mean" column
   # then will append the remaining columns
}
