% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/id.st.R
\name{id.st}
\alias{id.st}
\title{Identification of SVAR models by means of a smooth transition of volatility}
\usage{
id.st(x, c_lower = 0.3, c_upper = 0.7, c_step = 5, c_fix = NULL,
  transition_variable = NULL, gamma_lower = -3, gamma_upper = 2,
  gamma_step = 0.5, gamma_fix = NULL, nc = 1, max.iter = 5,
  crit = 0.001, restriction_matrix = NULL, lr_test = FALSE)
}
\arguments{
\item{x}{An object of class 'vars', 'vec2var', 'nlVar'. Estimated VAR object}

\item{c_lower}{Numeric. Starting point for the algorithm to start searching for the volatility shift.
Default is 0.3*(Total number of observations)}

\item{c_upper}{Numeric. Ending point for the algorithm to stop searching for the volatility shift.
Default is 0.7*(Total number of observations). Note that in case of a stochastic transition variable, the input requires an absolute value}

\item{c_step}{Integer. Step width of c. Default is 5. Note that in case of a stochastic transition variable, the input requires an absolute value}

\item{c_fix}{Numeric. If the transition point is known, it can be passed as an argument
where transition point = Number of observations - c_fix}

\item{transition_variable}{A numeric vector that represents the transition variable. By default (NULL), the time is used
as transition variable. Note that c_lower,c_upper, c_step and/or c_fix have to be adjusted
to the specified transition variable}

\item{gamma_lower}{Numeric. Lower bound for gamma. Small values indicate a flat transition function. Default is -3}

\item{gamma_upper}{Numeric. Upper bound for gamma. Large values indicate a steep transition function. Default is 2}

\item{gamma_step}{Numeric. Step width of gamma. Default is 0.5}

\item{gamma_fix}{Numeric. A fixed value for gamma, alternative to gamma found by the function}

\item{nc}{Integer. Number of processor cores
Note that the smooth transition model is computationally extremely demanding.}

\item{max.iter}{Integer. Number of maximum GLS iterations}

\item{crit}{Numeric. Critical value for the precision of the GLS estimation}

\item{restriction_matrix}{Matrix. A matrix containing presupposed entries for matrix B, NA if no restriction is imposed (entries to be estimated). Alternatively, a K^2*K^2 matrix can be passed, where ones on the diagonal designate unrestricted and zeros restricted coefficients. (as suggested in Luetkepohl, 2017, section 5.2.1).}

\item{lr_test}{Logical. Indicates whether the restricted model should be tested against the unrestricted model via a likelihood ratio test}
}
\value{
A list of class "svars" with elements
\item{Lambda}{Estimated heteroscedasticity matrix \eqn{\Lambda}}
\item{Lambda_SE}{Matrix of standard errors of Lambda}
\item{B}{Estimated structural impact matrix B, i.e. unique decomposition of the covariance matrix of reduced form residuals}
\item{B_SE}{Standard errors of matrix B}
\item{n}{Number of observations}
\item{Fish}{Observed Fisher information matrix}
\item{Lik}{Function value of likelihood}
\item{wald_statistic}{Results of pairwise Wald tests}
\item{iteration}{Number of GLS estimations}
\item{method}{Method applied for identification}
\item{est_c}{Structural break (number of observations)}
\item{est_g}{Transition coefficient}
\item{transition_variable}{Vector of transition variable}
\item{comb}{Number of all grid combinations of gamma and c}
\item{transition_function}{Vector of transition function}
\item{A_hat}{Estimated VAR parameter via GLS}
\item{type}{Type of the VAR model e.g., 'const'}
\item{y}{Data matrix}
\item{p}{Number of lags}
\item{K}{Dimension of the VAR}
\item{restrictions}{Number of specified restrictions}
\item{restriction_matrix}{Specified restriction matrix}
\item{lr_test}{Logical, whether a likelihood ratio test is performed}
\item{lRatioTest}{Results of likelihood ratio test}
}
\description{
Given an estimated VAR model, this function uses a smooth transition in the covariance to identify the structural impact matrix B of the corresponding SVAR model
\deqn{y_t=c_t+A_1 y_{t-1}+...+A_p y_{t-p}+u_t
=c_t+A_1 y_{t-1}+...+A_p y_{t-p}+B \epsilon_t.}
Matrix B corresponds to the decomposition of the pre-break covariance matrix \eqn{\Sigma_1=B B'}.
The post-break covariance corresponds to \eqn{\Sigma_2=B\Lambda B'} where \eqn{\Lambda} is the estimated heteroskedasticity matrix.
}
\examples{
\donttest{
# data contains quartlery observations from 1965Q1 to 2008Q2
# x = output gap
# pi = inflation
# i = interest rates
set.seed(23211)
v1 <- vars::VAR(USA, lag.max = 10, ic = "AIC" )
x1 <- id.st(v1, c_fix = 80, gamma_fix = 0)
summary(x1)
plot(x1)

# switching columns according to sign patter
x1$B <- x1$B[,c(3,2,1)]
x1$B[,3] <- x1$B[,3]*(-1)

# Impulse response analysis
i1 <- irf(x1, n.ahead = 30)
plot(i1, scales = 'free_y')

# Example with same data set as in Luetkepohl and Nestunajev 2017
v1 <- vars::VAR(LN, p = 3, type = 'const')
x1 <- id.st(v1, c_fix = 167, gamma_fix = -2.77)
summary(x1)
plot(x1)

}
}
\references{
Luetkepohl H., Netsunajev A., 2017. Structural vector autoregressions with smooth transition \cr
  in variances. Journal of Economic Dynamics and Control, 84, 43 - 57. ISSN 0165-1889.
}
\seealso{
For alternative identification approaches see \code{\link{id.cv}}, \code{\link{id.garch}}, \code{\link{id.cvm}}, \code{\link{id.dc}},
         or \code{\link{id.ngml}}
}
