% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/growth.R
\name{growth}
\alias{growth}
\alias{rolling_growth}
\title{Rolling basic growth}
\usage{
growth(x, y, na.rm = FALSE, log = FALSE, inf_fill = NULL)

rolling_growth(
  x,
  n = 1,
  lag = n,
  na.rm = FALSE,
  partial = TRUE,
  offset = NULL,
  weights = NULL,
  inf_fill = NULL,
  log = FALSE,
  ...
)
}
\arguments{
\item{x}{Numeric vector.}

\item{y}{numeric vector}

\item{na.rm}{Should missing values be removed when calculating window? Defaults to \code{FALSE}.}

\item{log}{If \code{TRUE} Growth (relative change) in total and mean events will be
calculated on the log-scale.}

\item{inf_fill}{Numeric value to replace \code{Inf} values with. Default behaviour is to keep \code{Inf} values.}

\item{n}{Rolling window size, default is 1.}

\item{lag}{Lag of basic growth comparison, default is the rolling window size.}

\item{partial}{Should rates be calculated outwith the window using partial windows?
If \code{TRUE} (the default), (n - 1) pairs of equally-sized rolling windows are compared,
their size increasing by 1 up to size n, at which point the rest of the window pairs are
all of size n. If \code{FALSE} all window-pairs will be of size n.}

\item{offset}{Numeric vector of values to use as offset, e.g. population sizes or exposure times.}

\item{weights}{Importance weights. These can either be
length 1 or the same length as x.
Currently, no normalisation of weights occurs.}

\item{...}{Further arguments to be passed on to \code{frollmean}.}
}
\value{
\code{growth} returns a \code{numeric(1)} and \code{rolling_growth}
returns a \code{numeric(length(x))}.
}
\description{
Calculate basic growth calculations on a rolling basis.
\code{growth()} calculates the percent change between the totals of two numeric vectors
when they're of equal length, otherwise the percent change between the means.
\code{rolling_growth()} does the same calculation on 1 numeric vector, on a rolling basis.
Pairs of windows of length \code{n}, lagged by the value specified by \code{lag} are compared in
a similar manner.
When \code{lag = n} then \code{data.table::frollsum()} is used,
otherwise \code{data.table::frollmean()} is used.
}
\examples{
library(timeplyr)
\dontshow{
.n_dt_threads <- data.table::getDTthreads()
.n_collapse_threads <- collapse::get_collapse()$nthreads
data.table::setDTthreads(threads = 1L)
collapse::set_collapse(nthreads = 1L)
}
set.seed(42)
# Growth rate is 6\% per day
x <- 10 * (1.06)^(0:25)

# Simple growth from one day to the next
rolling_growth(x, n = 1)

# Growth comparing rolling 3 day cumulative
rolling_growth(x, n = 3)

# Growth comparing rolling 3 day cumulative, lagged by 1 day
rolling_growth(x, n = 3, lag = 1)

# Growth comparing windows of equal size
rolling_growth(x, n = 3, partial = FALSE)

# Seven day moving average growth
roll_mean(rolling_growth(x), window = 7, partial = FALSE)
\dontshow{
data.table::setDTthreads(threads = .n_dt_threads)
collapse::set_collapse(nthreads = .n_collapse_threads)
}
}
