% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/query.R
\name{TrajExpectedSquareDisplacement}
\alias{TrajExpectedSquareDisplacement}
\title{Trajectory expected square displacement}
\usage{
TrajExpectedSquareDisplacement(trj, n = nrow(trj), eqn1 = TRUE,
  compass.direction = NULL)
}
\arguments{
\item{trj}{A Trajectory.}

\item{n}{Number of steps to calculate.}

\item{eqn1}{If \code{TRUE}, calculate using equation 1, otherwise using
equation 2. Equation 2 applies when the mean of turning angles is 0,
i.e.turns are unbiased.}

\item{compass.direction}{If not \code{NULL}, step angles are calculated
relative to this angle (in radians), otherwise they are calculated relative
to the previous step angle.}
}
\description{
Calculates the expected square displacement for a trajectory assuming it is a
correlated random walk, using the formula in Kareiva & Shigesada, (1983).
}
\details{
Note that Cheung, Zhang, Stricker, and Srinivasan (2007) define an
alternative formulation for expected maximum displacement, Emax (see
\code{\link{TrajEmax}}).
}
\examples{
set.seed(1)
# A random walk
trj <- TrajGenerate(200)
smoothed <- TrajSmoothSG(trj)

# Calculate actual squared displacement at all points along the trajectory
sd2 <- sapply(2:nrow(smoothed), function(n) TrajDistance(smoothed, 1, n) ^ 2)
# Calculate expected squared displacement
ed2_1 <- sapply(2:nrow(smoothed), function(n) TrajExpectedSquareDisplacement(smoothed, n, TRUE))
ed2_2 <- sapply(2:nrow(smoothed), function(n) TrajExpectedSquareDisplacement(smoothed, n, FALSE))

# Plot expected against actual. According to Kareiva & Shigesada, (1983), if actual
# (approximately) matches expected, the trajectory is probably a correlated random walk
par(mar = c(5, 5, 0.1, 0.1) + .1)
plot(2:nrow(smoothed), sd2, type = 'l', pch = 16, cex = .2, lwd = 2,
     xlab = 'Number of consecutive moves',
     ylab = expression('Squared displacement, ' * R[n]^2))
lines(2:nrow(smoothed), ed2_1, col = "grey", lwd = 2)
lines(2:nrow(smoothed), ed2_2, col = "pink", lwd = 2)

legend("bottomright",
       c(expression("Actual displacement"^2),
         expression("Expected displacement"^2 * " (eqn 1)"),
         expression("Expected displacement"^2 * " (eqn 2)")),
       col = c('black', 'grey', 'pink'), lwd = 2,
       inset = c(0.01, 0.02))

}
\references{
Cheung, A., Zhang, S., Stricker, C., & Srinivasan, M. V. (2007). Animal
navigation: the difficulty of moving in a straight line. Biological
Cybernetics, 97(1), 47-61. doi:10.1007/s00422-007-0158-0

Kareiva, P. M., & Shigesada, N. (1983). Analyzing insect movement as a
correlated random walk. Oecologia, 56(2), 234-238. doi:10.1007/bf00379695
}
\seealso{
\code{\link{TrajEmax}}
}
