% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ddd.R
\name{ddd}
\alias{ddd}
\title{Doubly Robust DDD estimators for the group-time average treatment effects.}
\usage{
ddd(
  yname,
  tname,
  idname,
  gname,
  pname,
  xformla,
  data,
  control_group = NULL,
  base_period = NULL,
  est_method = "dr",
  weightsname = NULL,
  boot = FALSE,
  nboot = NULL,
  cluster = NULL,
  cband = FALSE,
  alpha = 0.05,
  use_parallel = FALSE,
  cores = 1,
  inffunc = FALSE,
  skip_data_checks = FALSE
)
}
\arguments{
\item{yname}{The name of the outcome variable.}

\item{tname}{The name of the column containing the time periods.}

\item{idname}{The name of the column containing the unit id.}

\item{gname}{The name of the column containing the first period when a particular observation is treated. It is a positive number
for treated units and defines which group the unit belongs to. It takes value 0 or Inf for untreated units.}

\item{pname}{The name of the column containing the partition variable (e.g., the subgroup identifier). This is an indicator variable that is 1 for
the units eligible for treatment and 0 otherwise.}

\item{xformla}{The formula for the covariates to be included in the model. It should be of the form \code{~ x1 + x2}.
Default is \code{xformla = ~1} (no covariates).}

\item{data}{A data frame or data table containing the data.}

\item{control_group}{Valid for multiple periods only. The control group to be used in the estimation. Default is \code{control_group = "notyettreated"} which sets as control group the units that have not yet participated in the treatment.
The alternative is \code{control_group = "nevertreated"} which sets as control group the units that never participate in the treatment and does not change across groups or time periods.}

\item{base_period}{Valid for multiple periods. Choose between a "varying" or "universal" base period. Both yield the same post-treatment ATT(g,t) estimates.
Varying base period: Computes pseudo-ATT in pre-treatment periods by comparing outcome changes for a group to its comparison group from t-1 to t, repeatedly changing t.
Universal base period: Fixes the base period to (g-1), reporting average changes from t to (g-1) for a group relative to its comparison group, similar to event study regressions.
Varying base period reports ATT(g,t) right before treatment. Universal base period normalizes the estimate before treatment to be 0, adding one extra estimate in an earlier period.}

\item{est_method}{The estimation method to be used. Default is \code{"dr"} (doubly robust). It computes propensity score using logistic regression
and outcome regression using OLS. The alternative are \code{c("reg", "ipw")}.}

\item{weightsname}{The name of the column containing the weights. Default is \code{NULL}. As part of data processing, weights are enforced to be normalized
and have mean 1 across all observations.}

\item{boot}{Logical. If \code{TRUE}, the function computes standard errors using the multiplier bootstrap. Default is \code{FALSE}.}

\item{nboot}{The number of bootstrap samples to be used. Default is \code{NULL}. If \code{boot = TRUE}, the default is \code{nboot = 999}.}

\item{cluster}{The name of the variable to be used for clustering. The maximum number of cluster variables is 1. Default is \code{NULL}.
If \code{boot = TRUE}, the function computes the bootstrap standard errors clustering at the unit level setting as cluster variable the one in \code{idname}.}

\item{cband}{Logical. If \code{TRUE}, the function computes a uniform confidence band that covers all of the average treatment effects
with fixed probability \code{1-alpha}.  In order to compute uniform confidence bands, \code{boot} must also be set to \code{TRUE}.  The default is \code{FALSE}.}

\item{alpha}{The level of significance for the confidence intervals.  Default is \code{0.05}.}

\item{use_parallel}{Logical. If \code{TRUE}, the function runs in parallel processing. Valid only when \code{boot = TRUE}. Default is \code{FALSE}.}

\item{cores}{The number of cores to be used in the parallel processing. Default is \code{cores = 1}.}

\item{inffunc}{Logical. If \code{TRUE}, the function returns the influence function. Default is \code{FALSE}.}

\item{skip_data_checks}{Logical. If \code{TRUE}, the function skips the data checks and go straight to estimation. Default is \code{FALSE}.}
}
\value{
A \code{ddd} object with the following basic elements:
\item{ATT}{The average treatment effect on the treated.}
\item{se}{The standard error of the ATT.}
\item{uci}{The upper confidence interval of the ATT.}
\item{lci}{The lower confidence interval of the ATT.}
\item{inf_func}{The estimate of the influence function.}
}
\description{
\code{ddd} is the main function for computing the Doubly Robust DDD estimators for the ATT, with balanced panel data.
It can be used with covariates and/or under multiple time periods. At its core, \code{triplediff} employs
the doubly robust estimator for the ATT, which is a combination of the propensity score weighting and the outcome regression.
Furthermore, this package supports the application of machine learning methods for the estimation of the nuisance parameters.
}
\examples{
#----------------------------------------------------------
# Triple Diff with covariates and 2 time periods
#----------------------------------------------------------
set.seed(1234) # Set seed for reproducibility
# Simulate data for a two-periods DDD setup
df <- gen_dgp_2periods(size = 5000, dgp_type = 1)$data

head(df)

att_22 <- ddd(yname = "y", tname = "time", idname = "id", gname = "state",
              pname = "partition", xformla = ~cov1 + cov2 + cov3 + cov4,
             data = df, control_group = "nevertreated", est_method = "dr")

summary(att_22)



#----------------------------------------------------------
# Triple Diff with multiple time periods
#----------------------------------------------------------
data <- gen_dgp_mult_periods(size = 1000, dgp_type = 1)[["data"]]

ddd(yname = "y", tname = "time", idname = "id",
     gname = "state", pname = "partition", xformla = ~cov1 + cov2 + cov3 + cov4,
     data = data, control_group = "nevertreated", base_period = "varying",
     est_method = "dr")

}
