% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dist_profile.R
\name{dist_profile}
\alias{dist_profile}
\title{Calculates the distance profile using MASS algorithms}
\usage{
dist_profile(data, query, ..., window_size = NULL, method = "v3",
  index = 1, k = NULL, weight = NULL, paa = 1)
}
\arguments{
\item{data}{a \code{matrix} or a \code{vector}.}

\item{query}{a \code{matrix} or a \code{vector}. See details.}

\item{...}{Precomputed values from the first iteration. If not supplied, these values will be computed.}

\item{window_size}{an \code{int} or \code{NULL}. Sliding window size. See details.}

\item{method}{method that will be used to calculate the distance profile. See details.}

\item{index}{an \code{int}. Index of query window. See details.}

\item{k}{an \code{int} or \code{NULL}. Default is \code{NULL}. Defines the size of batch for MASS V3. Prefer to
use a power of 2. If \code{NULL}, it will be set automatically.}

\item{weight}{a \code{vector} of \code{numeric} or \code{NULL} with the same length of the \code{window_size}. This is
a MASS extension to weight the query.}

\item{paa}{a \code{numeric}. Default is \code{1}. Factor of PAA reduction (2 == half of size). This is a
MASS extension.}
}
\value{
Returns the \code{distance_profile} for the given query and the \code{last_product} for STOMP
algorithm and the parameters for recursive call. See details.
}
\description{
Mueen's Algorithm for Similarity Search is The Fastest Similarity Search Algorithm for Time
Series Subsequences under Euclidean Distance and Correlation Coefficient.
}
\details{
This function has several ways to work:

Case 1: You have a small sized query and the data. In this case you only have to provide the first two
parameters \code{data} and \code{query}. Internally the \code{window_size} will be get from the query length.

Case 2: You have one or two data vectors and want to compute the join or self-similarity. In this case
you need to use the recursive solution. The parameters are \code{data}, \code{query}, \code{window_size} and \code{index}.
The first iteration don't need the \code{index} unless you are starting somewhere else. The \code{query} will be
the source of a \code{query_window}, starting on \code{index}, with length of \code{window_size}.

The \code{method} defines which MASS will be used. Current supported values are: \code{v2}, \code{v3}, \code{weighted}.
}
\examples{

w <- mp_toy_data$sub_len
ref_data <- mp_toy_data$data[, 1]
# minimum example, data and query
nn <- dist_profile(ref_data, ref_data[1:w])
distance_profile <- Re(sqrt(nn$distance_profile))

# data and indexed query
nn <- dist_profile(ref_data, ref_data, window_size = w, index = 10)
distance_profile <- Re(sqrt(nn$distance_profile))

# recursive
nn <- NULL

for (i in seq_len(10)) {
  nn <- dist_profile(ref_data, ref_data, nn, window_size = w, index = i)
}

# weighted
weight <- c(rep(1, w / 3), rep(0.5, w / 3), rep(0.8, w / 3)) # just an example

nn <- dist_profile(ref_data, ref_data, window_size = w, index = 1, method = "weighted",
 weight = weight)
distance_profile <- Re(sqrt(nn$distance_profile))
}
\references{
\itemize{
\item Abdullah Mueen, Yan Zhu, Michael Yeh, Kaveh Kamgar, Krishnamurthy Viswanathan,
Chetan Kumar Gupta and Eamonn Keogh (2015), The Fastest Similarity Search Algorithm for Time
Series Subsequences under Euclidean Distance
}

Website: \url{https://www.cs.unm.edu/~mueen/FastestSimilaritySearch.html}
}
