% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/optimize_gps.R
\name{optimize_gps}
\alias{optimize_gps}
\title{Optimize the Matching Process via Random Search}
\usage{
optimize_gps(
  data = NULL,
  formula,
  ordinal_treat = NULL,
  n_iter = 1000,
  n_cores = 1,
  opt_args = NULL
)
}
\arguments{
\item{data}{A \code{data.frame} containing all variables specified in the
\code{formula} argument. If \code{opt_args} is used, the \code{data} provided within
\code{opt_args} must match this input exactly.}

\item{formula}{A valid formula object used to estimate the generalized
propensity scores (GPS). The treatment variable appears on the left-hand
side, and covariates on the right-hand side. Interactions can be specified
using \code{*}. See \code{\link[stats:formula]{stats::formula()}} and \code{\link[=estimate_gps]{estimate_gps()}} for more details. If
\code{opt_args} is provided, the formula within it must be identical to this
argument.}

\item{ordinal_treat}{An atomic vector defining the ordered levels of the
treatment variable. This confirms the variable is ordinal and adjusts its
levels accordingly using
\code{factor(treat, levels = ordinal_treat, ordered = TRUE)}. It is passed
directly to \code{estimate_gps()}. If \code{NULL}, ordinal GPS estimation methods
such as \code{polr} will be excluded from the optimization. See \code{\link[=estimate_gps]{estimate_gps()}}
for details.}

\item{n_iter}{Integer. Number of unique parameter combinations to evaluate
during optimization. Higher values generally yield better results but
increase computation time. For large datasets or high-dimensional parameter
spaces, increasing \code{n_iter} is recommended. When using parallel processing
(\code{n_cores > 1}), performance gains become more apparent with larger
\code{n_iter}. Too many cores with too few iterations may introduce
overhead and reduce efficiency.}

\item{n_cores}{Integer. Number of CPU cores to use for parallel execution. If
set to a value greater than 1, a parallel backend is registered using
\code{\link[future:multisession]{future::multisession()}}. \strong{Note:} parallel execution can significantly
increase memory usage. With large datasets or high \code{n_iter} values, RAM
consumption may spike, especially on systems with 16-32 GB RAM. Users are
advised to monitor system resources. Internally, the function performs
memory cleanup post-execution to manage resource usage efficiently.}

\item{opt_args}{An object of class \code{"opt_args"} containing optimization
parameters and argument settings. Use \code{\link[=make_opt_args]{make_opt_args()}} to create this
object. It specifies the search space for the GPS estimation and matching
procedure.}
}
\value{
An S3 object of class \code{best_opt_result}. The core component is a
\code{data.frame} containing the parameter combinations and results of the
optimization procedure. You can access it using \code{attr(result, "opt_results")} or by calling \code{View(result)}, where \code{result} is your
\code{best_opt_result} object.

The object contains the following custom attributes:
\itemize{
\item \strong{\code{opt_results}}: A \code{data.frame} of optimization results. Each row corresponds to a unique parameter combination tested. For a complete description of columns, see the \emph{\strong{Details}} section.
\item \strong{\code{optimization_time}}: Time (in seconds) taken by the optimization loop (i.e., the core \code{for}-loop that evaluates combinations). This does \strong{not} include the time needed for GPS estimation, pre-processing, or merging of results after loop completion. On large datasets, these excluded steps can still be substantial.
\item \strong{\code{combinations_tested}}: Total number of unique parameter combinations evaluated during optimization.
\item \strong{\code{smd_results}}: A detailed table of standardized mean differences (SMDs) for all pairwise treatment group comparisons and for all covariates specified in the \code{formula}. This is used by the \code{\link[=select_opt]{select_opt()}} function to filter optimal models based on covariate-level balance across groups.
\item \strong{\code{treat_names}}: A character vector with the names of the unique treatment groups.
\item \strong{\code{model_covs}}: A character vector listing the model covariates (main effects and interactions) used in the \code{formula}. These names correspond to the variables shown in the \code{smd_results} table.
}
}
\description{
The \code{optimize_gps()} function performs a random search to
identify optimal combinations of parameters for the \code{match_gps()} and
\code{estimate_gps()} functions. The goal is to maximize the percentage of
matched samples (\code{perc_matched}) while minimizing the maximum standardized
mean difference (\code{smd}), thereby improving the overall balance of
covariates across treatment groups. The function supports parallel
execution through the \code{foreach} and \code{future} packages, enabling
multithreaded computation to accelerate the optimization process,
particularly when dealing with large datasets or complex parameter spaces.
}
\details{
The output is an S3 object of class \code{best_opt_result}. Its core
component is a \code{data.frame} containing the parameter settings for the
best-performing models, grouped and ranked based on their balance quality.

Optimization results are categorized into seven bins based on the maximum
standardized mean difference (SMD):
\itemize{
\item 0.00-0.05
\item 0.05-0.10
\item 0.10-0.15
\item 0.15-0.20
\item 0.20-0.25
\item 0.25-0.30
\item Greater than 0.30
}

Within each SMD group, the parameter combination(s) achieving the highest
\code{perc_matched} (i.e., percentage of matched samples) is selected. In cases
where multiple combinations yield identical \code{smd} and \code{perc_matched}, all
such results are retained. Combinations where matching failed or GPS
estimation did not converge will return \code{NA} in the result columns (e.g.,
\code{perc_matched}, \code{smd}).

The returned \code{data.frame} includes the following columns (depending on the
number of treatment levels):
\itemize{
\item \code{iter_ID}: Unique identifier for each parameter combination
\item \code{method_match}: Matching method used in \code{\link[=match_gps]{match_gps()}}, e.g., \code{"nnm"} or \code{"fullopt"}
\item \code{caliper}: Caliper value used in \code{\link[=match_gps]{match_gps()}}
\item \code{order}: Ordering of GPS scores prior to matching
\item \code{kmeans_cluster}: Number of k-means clusters used
\item \code{replace}: Whether replacement was used in matching (\code{nnm} only)
\item \code{ties}: Tie-breaking rule in nearest-neighbor matching (\code{nnm} only)
\item \code{ratio}: Control-to-treated ratio for \code{nnm}
\item \code{min_controls}, \code{max_controls}: Minimum and maximum controls for \code{fullopt}
\item \code{reference}: Reference group used in both \code{\link[=estimate_gps]{estimate_gps()}} and \code{\link[=match_gps]{match_gps()}}
\item \code{perc_matched}: Percentage of matched samples (from \code{\link[=balqual]{balqual()}})
\item \code{smd}: Maximum standardized mean difference (from \code{\link[=balqual]{balqual()}})
\item \verb{p_\{group_name\}}: Percent matched per treatment group (based on group sample size)
\item \code{method_gps}: GPS estimation method used (from \code{\link[=estimate_gps]{estimate_gps()}})
\item \code{link}: Link function used in GPS model
\item \code{smd_group}: SMD range category for the row
}

The resulting \code{best_opt_result} object also includes a custom \code{print()}
method that summarizes:
\itemize{
\item The number of optimal parameter sets per SMD group
\item Their associated SMD and match rates
\item Total combinations tested
\item Total runtime of the optimization loop
}
}
\examples{
# Define formula for GPS estimation and matching
formula_cancer <- formula(status ~ age * sex)

# Set up the optimization parameter space
opt_args <- make_opt_args(cancer, formula_cancer, gps_method = "m1")

# Run optimization with 2000 random parameter sets and a fixed seed
\dontrun{
withr::with_seed(
  8252,
  {
    optimize_gps(
      data = cancer,
      formula = formula_cancer,
      opt_args = opt_args,
      n_iter = 2000
    )
  }
)
}
}
