% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/generateSpFromBCA.R
\name{generateSpFromBCA}
\alias{generateSpFromBCA}
\title{Generate a virtual species distribution from a Between Component Analysis of environmental variables}
\usage{
generateSpFromBCA(raster.stack.current, raster.stack.future, rescale = TRUE,
  niche.breadth = "any", means = NULL, sds = NULL, bca = NULL,
  sample.points = FALSE, nb.points = 10000, plot = TRUE)
}
\arguments{
\item{raster.stack.current}{a RasterStack object, in which each layer represent an environmental 
variable from the "current" time horizon.}

\item{raster.stack.future}{a RasterStack object, in which each layer represent an environmental 
variable from a "future" time horizon.}

\item{rescale}{\code{TRUE} of \code{FALSE}. Should the output suitability raster be
rescaled between 0 and 1?}

\item{niche.breadth}{\code{"any"}, \code{"narrow"} or \code{"wide"}. This parameter
defines how tolerant is the species regarding environmental conditions by adjusting
the standard deviations of the gaussian functions. See details.}

\item{means}{a vector containing two numeric values. Will be used to define
the means of the gaussian response functions to the axes of the BCA.}

\item{sds}{a vector containing two numeric values. Will be used to define
the standard deviations of the gaussian response functions to the axes of 
the BCA.}

\item{bca}{a \code{bca} object. You can provide a bca object that you 
already computed yourself with \code{\link[virtualspecies]{generateSpFromBCA}}}

\item{sample.points}{\code{TRUE} of \code{FALSE}. If you have large
raster files then use this parameter to sample a number of points equal to
\code{nb.points}. However the representation of your environmental variables will not be complete.}

\item{nb.points}{a numeric value. Only useful if \code{sample.points = TRUE}.
The number of sampled points from the raster, to perform the PCA. A too small
value may not be representative of the environmental conditions in your rasters.}

\item{plot}{\code{TRUE} or \code{FALSE}. If \code{TRUE}, the generated virtual species will be plotted.}
}
\value{
a \code{list} with 4 elements:
\itemize{
\item{\code{approach}: the approach used to generate the species, \emph{i.e.}, \code{"bca"}}
\item{\code{details}: the details and parameters used to generate the species}
\item{\code{suitab.raster.current}: the virtual species distribution, as a Raster object containing the
current environmental suitability}
\item{\code{suitab.raster.future}: the virtual species distribution, as a Raster object containing the
future environmental suitability}
}
The structure of the virtualspecies object can be seen using str()
}
\description{
A Between Component Analysis is similar to a PCA, except that two sets of environmental conditions 
(e.g. current and future) will be used. This function is useful to generate species in 
no-analogue climates.
}
\details{
This function generates a virtual species distribution by computing a Between
Component Analysis based on two different stacks of environmental variables.
The response of the species is then simulated along the two first axes of the BCA with gaussian functions in the
same way as in \code{\link{generateSpFromPCA}}.

A Between Component Analysis is used to separate two sets of environmental conditions.
This function proceeds in 4 steps:
\enumerate{
\item{A Principal Component Analysis is generated based on both set of environmental conditions}
\item{A BCA of this PCA is generated using the function \code{\link[ade4:bca]{bca}}
from package \code{ade4}. Note that at this step we choose  one random point
from \code{raster.stack.future},
and we use this single point as if it was a third set of environmental
conditions for the BCA. This trick allows us to subtly change the shape of the bca in order to
generate different types of conditions.}
\item{Gaussian responses to the first two axes are computed}
\item{These responses are multiplied to obtain the final environmental suitability}}

If \code{rescale = TRUE}, the final environmental suitability is rescaled between 0 and 1,
with the formula (val - min) / (max - min).

The shape of gaussian responses can be randomly generated by the function or defined manually by choosing
\code{means} and \code{sds}. The random generation is constrained
by the argument \code{niche.breadth}, which controls the range of possible 
standard deviation values. This range of values is based on
a fraction of the axis:
\itemize{
\item{\code{"any"}: the standard deviations can have values from 1\% to 50\% of axes' ranges. For example if the first axis of the PCA ranges from -5 to +5,
then sd values along this axe can range from 0.1 to 5.
}
\item{\code{"narrow"}: the standard deviations are limited between 1\% and 10\% of axes' ranges. For example if the first axis of the PCA ranges from -5 to +5,
then sd values along this axe can range from 0.1 to 1.
}
\item{\code{"wide"}: the standard deviations are limited between 10\% and 50\% of axes' ranges. For example if the first axis of the PCA ranges from -5 to +5,
then sd values along this axe can range from 1 to 5.
}
}
If a \code{bca} object is provided, the output bca object will contain the new environments coordinates along the provided bca axes.
}
\note{
To perform the BCA, the function has to transform rasters into matrices.
This may not be feasible if the chosen rasters are too large for the computer's memory.
In this case, you should run the function with \code{sample.points = TRUE} 
and set the number of points to sample with \code{nb.points}.
}
\examples{
# Create two example stacks with four environmental variables each
a <- matrix(rep(dnorm(1:100, 50, sd = 25)), 
            nrow = 100, ncol = 100, byrow = TRUE)

env1 <- stack(raster(a * dnorm(1:100, 50, sd = 25)),
              raster(a * 1:100),
              raster(a),
              raster(t(a)))
names(env1) <- c("var1", "var2", "var3", "var4")
plot(env1) # Illustration of the variables

b <- matrix(rep(dnorm(1:100, 25, sd = 50)), 
            nrow = 100, ncol = 100, byrow = TRUE)

env2 <- stack(raster(b * dnorm(1:100, 50, sd = 25)),
              raster(b * 1:100),
              raster(b),
              raster(t(b)))

names(env2) <- c("var1", "var2", "var3", "var4")
plot(env2) # Illustration of the variables 

# Generating a species with the BCA

generateSpFromBCA(raster.stack.current = env1, raster.stack.future = env2)

# The left part of the plot shows the BCA and the response functions along
# the two axes.
# The top-right part shows environmental suitability of the virtual
# species in the current environment.
# The bottom-right part shows environmental suitability of the virtual
# species in the future environment. 


# Defining manually the response to axes

generateSpFromBCA(raster.stack.current = env1, raster.stack.future = env2,
           means = c(-2, 0),
           sds = c(0.6, 1.5))
   
                   
}
\seealso{
\code{\link{generateSpFromFun}} to generate a virtual species with
the responses to each environmental variables.\code{\link{generateSpFromPCA}} to generate a virtual species with
the PCA of environmental variables.
}
\author{
Robin Delsol, Boris Leroy

Maintainer: Boris Leroy \email{leroy.boris@gmail.com}
}
