##
## wdnet: Weighted directed network
## Copyright (C) 2023  Yelie Yuan, Tiandong Wang, Jun Yan and Panpan Zhang
## Jun Yan <jun.yan@uconn.edu>
##
## This file is part of the R package wdnet.
##
## The R package wdnet is free software: You can redistribute it and/or
## modify it under the terms of the GNU General Public License as published by
## the Free Software Foundation, either version 3 of the License, or any later
## version (at your option). See the GNU General Public License at
## <https://www.gnu.org/licenses/> for details.
##
## The R package wdnet is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
##

#' @importFrom stats cor
#' @importFrom CVXR norm2
#' @importFrom utils modifyList
NULL

#' Degree preserving rewiring for directed networks
#' 
#' Degree preserving rewiring towards the target structure \code{eta}.
#'
#' @param edgelist A two column matrix, each row represents a directed edge from
#'   the first column to the second column.
#' @param eta An matrix, target structure eta generated by
#'   \code{wdnet::get_eta_directed()}.
#' @param iteration An integer, number of rewiring iterations, each iteration
#'   consists of \code{nattempts} rewiring attempts.
#' @param nattempts An integer, number of rewiring attempts for each iteration.
#'   Default value equals the number of rows of edgelist.
#' @param rewire.history Logical, whether the rewiring history should be
#'   returned.
#'
#' @return Rewired edgelist, degree based assortativity coefficients after each
#'   iteration, rewiring history (including the index of sampled edges and
#'   rewiring result). For each rewiring attempt, two rows are sampled form the
#'   edgelist, for example Edge1:(v_1, v_2) and Edge2:(v_3, v_4), if the
#'   rewiring attempt is accepted, the sampled edges are replaced as (v_1, v_4),
#'   (v_3, v_2).
#' 
#' @keywords internal
#'
dprewire_directed <- function(edgelist, eta, 
                              iteration = 200, nattempts = NULL, 
                              rewire.history = FALSE) {
  if (is.null(nattempts)) nattempts <- nrow(edgelist)
  edgelist <- as.matrix(edgelist)
  sourceNode <- edgelist[, 1]
  targetNode <- edgelist[, 2]
  temp <- node_strength_cpp(snode = sourceNode, 
                            tnode = targetNode, 
                            nnode = max(edgelist), 
                            weight = 1,
                            weighted = FALSE)
  outd <- temp$outstrength
  ind <- temp$instrength
  
  sourceOut <- outd[sourceNode]
  sourceIn <- ind[sourceNode]
  targetOut <- outd[targetNode]
  targetIn <- ind[targetNode]
  
  df_s <- data.frame(type = rownames(eta), 
                     index = seq_len(nrow(eta)) - 1)
  df_t <- data.frame(type = colnames(eta), 
                     index = seq_len(ncol(eta)) - 1)
  type_s <- paste0(sourceOut, "-", sourceIn, split = "")
  type_t <- paste0(targetOut, "-", targetIn, split = "")
  
  index_s <- df_s[match(type_s, df_s$type), "index"]
  index_t <- df_t[match(type_t, df_t$type), "index"]
  rm(df_s, df_t, type_s, type_t, temp, outd, ind)
  
  ret <- dprewire_directed_cpp(iteration, nattempts, 
                               targetNode, 
                               sourceOut, sourceIn,
                               targetOut, targetIn,
                               index_s, index_t, 
                               eta, rewire.history)
  rho <- data.frame("Iteration" = c(0:iteration), 
                    "outout" = NA, 
                    "outin" = NA, 
                    "inout" = NA, 
                    "inin" = NA)
  rho[1, 2:5] <- c("outout" = stats::cor(sourceOut, targetOut), 
                   "outin" = stats::cor(sourceOut, targetIn), 
                   "inout" = stats::cor(sourceIn, targetOut),
                   "inin" = stats::cor(sourceIn, targetIn))
  rho[2:(iteration + 1), 2] <- ret$out_out
  rho[2:(iteration + 1), 3] <- ret$out_in
  rho[2:(iteration + 1), 4] <- ret$in_out
  rho[2:(iteration + 1), 5] <- ret$in_in
  
  colnames(rho) <- c("Iteration", "outout", "outin", "inout", "inin")
  edgelist[, 2] <- ret$targetNode
  result <- list("assortcoef" = rho, 
                 "edgelist" = edgelist,
                 "iteration" = iteration,
                 "nattempts" = nattempts)
  if (rewire.history) {
    colnames(ret$history) <- c("Attempt", "Edge1", "Edge2", "Accepted")
    ret$history[, 1:3] <- ret$history[, 1:3] + 1
    result$history <- ret$history
  }
  return(result)
}

#' Degree preserving rewiring for undirected networks
#' 
#' Degree preserving rewiring towards the target structure \code{eta}.
#'
#' @param edgelist A two column matrix, each row represents an undirected edge.
#' @param iteration An integer, number of rewiring iterations, each iteration
#'   consists of \code{nattempts} rewiring attempts.
#' @param nattempts An integer, number of rewiring attempts for each iteration.
#'   Default value equals the number of rows of edgelist.
#' @param eta An matrix, target structure \code{eta} generated by
#'   \code{wdnet::get_eta_undirected()}.
#' @param rewire.history Logical, whether the rewiring history should be
#'   returned.
#' @return Rewired edgelist, assortativity coefficient after each iteration, and
#'   rewiring history (including the index of sampled edges and rewiring
#'   result). For each rewiring attempt, two rows are sampled from the edgelist,
#'   for example Edge1:\{v_1, v_2\} and Edge2:\{v_3, v_4\}, the function try to 
#'   rewire the sampled edges as \{v_1, v_4\}, \{v_3, v_2\} (rewire type 1) 
#'   or \{v_1, v_3\}, \{v_2, v_4\} (rewire type 2) with probability 1/2.
#' 
#' @keywords internal
#'
dprewire_undirected <- function(edgelist, eta, 
                                iteration = 200, nattempts = NULL, 
                                rewire.history = FALSE) {
  if (is.null(nattempts)) nattempts <- nrow(edgelist)
  
  edgelist <- as.matrix(edgelist)
  degree <- data.frame(table(c(edgelist)))$Freq
  d_df <- data.frame(type = rownames(eta), index = seq_len(nrow(eta)) - 1)
  node1 <- edgelist[, 1]
  node2 <- edgelist[, 2]
  index1 <- d_df[match(degree[node1], d_df$type), "index"]
  index2 <- d_df[match(degree[node2], d_df$type), "index"]
  rm(d_df)
  degree1 <- degree[c(node1, node2)]
  degree2 <- degree[c(node2, node1)]
  ret <- dprewire_undirected_cpp(iteration, nattempts, 
                                 node1, node2,
                                 degree1, degree2,
                                 index1, index2,
                                 eta, rewire.history)
  rm(node1, node2, degree1, degree2, index1, index2)
  rho <- data.frame("Iteration" = c(0:iteration), "Value" = NA)
  rho[1, 2] <- assortcoef(edgelist, directed = FALSE)
  rho[2:(iteration + 1), 2] <- ret$rho
  colnames(rho) <- c("Iteration", "Value")
  edgelist <- cbind(ret$node1, ret$node2)
  result <- list("assortcoef" = rho,
                 "edgelist" = edgelist,
                 "iteration" = iteration,
                 "nattempts" = nattempts)
  if (rewire.history) {
    colnames(ret$history) <- c("Attempt", "Edge1", "Edge2", "RewireType", "Accepted")
    ret$history[, 1:4] <- ret$history[, 1:4] + 1
    result$history <- ret$history
  }
  return(result)
}

#' Degree preserving rewiring.
#'
#' Rewire a given network to have predetermined assortativity coefficients while
#' preserving node degree.
#'
#' There are two steps in this algorithm. It first solves for an appropriate
#' \code{eta} using \code{target.assortcoef}, \code{eta.obj}, and
#' \code{cvxr_control}, then proceeds to the rewiring process and rewire the
#' network towards the solved \code{eta}. If \code{eta} is given, the algorithm
#' will skip the first step. The function only works for unweighted networks.
#'
#' Each rewiring attempt samples two rows from \code{edgelist}, for example
#' Edge1:(v_1, v_2) and Edge2:(v_3, v_4). For directed networks, if the rewiring
#' attempt is accepted, the sampled edges are replaced as (v_1, v_4), (v_3,
#' v_2); for undirected networks, the algorithm try to rewire the sampled edges 
#' as \{v_1, v_4\}, \{v_3, v_2\} (type 1) or \{v_1, v_3\}, \{v_2, v_4\} 
#' (type 2), each with probability 1/2.
#'
#' @param edgelist A two column matrix, each row represents an edge of the
#'   network.
#' @param directed Logical, whether the network is directed or not.
#' @param adj Adjacency matrix of an unweighted network. It will be ignored if
#'   \code{edgelist} is provided.
#' @param target.assortcoef For directed networks, it is a list represents the
#'   predetermined value or range of assortativity coefficients. For undirected
#'   networks, it is a constant between -1 to 1. It will be ignored if
#'   \code{eta} is provided.
#' @param control A list of parameters for controlling the rewiring process and
#'   the process for solving \code{eta}. \itemize{ \item{\code{iteration}} {An
#'   integer, represents the number of rewiring iterations. Each iteration
#'   consists of \code{nattempts} rewiring attempts. The assortativity
#'   coefficient(s) of the network will be recorded after each iteration.}
#'   \item{\code{nattempts}} {An integer, number of rewiring attempts for each
#'   iteration. Default value equals the number of rows of \code{edgelist}}.
#'   \item{\code{history}} {Logical, whether the rewiring attempts should be
#'   recorded and returned.} \item{\code{eta.obj}} {A convex function of
#'   \code{eta} to be minimized when solving for \code{eta} with given
#'   \code{target.assortcoef}. Defaults to 0. It will be ignored if \code{eta}
#'   is provided.} \item{\code{cvxr_control} {A list of parameters passed to
#'   \code{CVXR::solve()} for solving \code{eta} with given
#'   \code{target.assortcoef}. It will be ignored if \code{eta} is provided.}}}
#' @param eta An matrix represents the target network structure. If specified,
#'   \code{target.assortcoef} will be ignored. For directed networks, the
#'   element at row "i-j" and column "k-l" represents the proportion of directed
#'   edges linking a source node with out-degree i and in-degree j to a target
#'   node with out-degree k and in-degree l. For undirected networks, \code{eta}
#'   is symmetric, the summation of the elements at row "i", column "j" and row
#'   "j", column "i" represents the proportion of edges linking to a node with
#'   degree i and a node with degree j.
#'
#' @return Rewired \code{edgelist}; assortativity coefficient(s) after each
#'   iteration; rewiring history (including the index of sampled edges and
#'   rewiring result) and solver results.
#'
#' @export
#'
#' @examples
#' \donttest{
#' set.seed(123)
#' edgelist <- rpanet(1e4, control = rpa_control_scenario(
#'    alpha = 0.4, beta = 0.3, gamma = 0.3))$edgelist
#' ## rewire a directed network to have predetermined assortativity coefficients
#' target.assortcoef <- list("outout" = -0.2, "outin" = 0.2)
#' ret1 <- dprewire(edgelist, directed = TRUE,
#'                target.assortcoef = target.assortcoef,
#'                control = list(iteration = 200))
#' plot(ret1$assortcoef$Iteration, ret1$assortcoef$"outout")
#' plot(ret1$assortcoef$Iteration, ret1$assortcoef$"outin")
#'
#' edgelist <- rpanet(1e4, control = rpa_control_scenario(
#'                    alpha = 0.3, beta = 0.1, gamma = 0.3, xi = 0.3),
#'                    directed = FALSE)$edgelist
#' ## rewire an undirected network to have predetermined assortativity coefficient
#' ret2 <- dprewire(edgelist, directed = FALSE, target.assortcoef = 0.3,
#'                control = list(iteration = 300, eta.obj = CVXR::norm2, 
#'                history = TRUE))
#' plot(ret2$assortcoef$Iteration, ret2$assortcoef$Value)
#' }
#' 

dprewire <- function(edgelist = NULL, directed = TRUE, adj = NULL,
                     target.assortcoef = list("outout" = NULL, 
                                              "outin" = NULL, 
                                              "inout" = NULL, 
                                              "inin" = NULL),
                     control = list("iteration" = 200, 
                                    "nattempts" = NULL, 
                                    "history" = FALSE, 
                                    "cvxr_control" = cvxr_control(),
                                    "eta.obj" = function(x) 0),
                     eta = NULL) {
  if (is.null(edgelist)) {
    if (is.null(adj)) {
      stop('"edgelist" and "adj" can not both be NULL.')
    }
    stopifnot('"dprewire" only works for unweighted networks.' = 
                all(adj %% 1 == 0))
    if (! all(adj == 1)) {
      warning('The elements of "adj" are used as the number of edges between nodes.')
    }
    temp <- adj_to_edge(adj = adj, directed = directed, weighted = NULL)
    edgelist <- temp$edgelist
    rm(temp)
  }
  
  stopifnot("Nodes must be consecutive integers starting from 1." = 
              min(edgelist) == 1 & max(edgelist) == length(unique(c(edgelist))))
  
  control.default <- list("iteration" = 200, 
                          "nattempts" = NULL, 
                          "history" = FALSE, 
                          "cvxr_control" = cvxr_control(),
                          "eta.obj" = function(x) 0)
  control <- utils::modifyList(control.default, control, keep.null = TRUE)
  rm(control.default)
  
  solver.result <- NULL
  if (is.null(eta)) {
    if (directed) {
      solver.result <- get_eta_directed(edgelist = edgelist,
                                        target.assortcoef = target.assortcoef,
                                        eta.obj = control$eta.obj, 
                                        control = control$cvxr_control)
    }
    else {
      stopifnot('"target.assortcoef" must be a constant between -1 and 1 if the network is undirected.' = 
                  is.numeric(target.assortcoef) & target.assortcoef >= -1 & 
                  target.assortcoef <= 1)
      solver.result <- get_eta_undirected(edgelist = edgelist,
                                          target.assortcoef = target.assortcoef,
                                          eta.obj = control$eta.obj, 
                                          control = control$cvxr_control)
    }
    if (solver.result$status == "solver_error" | solver.result$status == "infeasible") {
      return(list("solver.result" = solver.result))
    }
    eta <- solver.result$eta
  }
  if (directed) {
    ret <- dprewire_directed(edgelist = edgelist,
                             eta = eta, 
                             iteration = control$iteration,
                             nattempts = control$nattempts,
                             rewire.history = control$history)
  }
  else {
    ret <- dprewire_undirected(edgelist = edgelist,
                               eta = eta, 
                               iteration = control$iteration,
                               nattempts = control$nattempts,
                               rewire.history = control$history)
  }
  ret$"solver.result" <- solver.result
  ret
}

#' Range of assortativity coefficient.
#'
#' The assortativity coefficient of a given network may not achieve all the
#' values within -1 and 1 via degree preserving rewiring. This function computes
#' the range of assortativity coefficients that can be achieved through degree
#' preserving rewiring. The algorithm is designed for unweighted networks.
#'
#' The ranges are computed through convex optimization. The problems are defined
#' and solved via the \code{R} package \code{CVXR}. For undirected networks, the
#' function returns the range of the assortativity coefficient. For directed
#' networks, the function computes the range of \code{which.range} while other
#' assortativity coefficients are restricted through \code{target.assortcoef}.
#' 
#' @param edgelist A two column matrix, each row represents an edge of the
#'   network.
#' @param directed Logical, whether the network is directed or not.
#' @param adj Adjacency matrix of an unweighted network. It will be ignored if
#'   \code{edgelist} is provided.
#' @param which.range The type of interested assortativity coefficient. For
#'   directed networks, it takes one of the values: "outout", "outin", "inout"
#'   and "inin". It will be ignored if the network is undirected.
#' @param target.assortcoef A list of constraints, it has the predetermined 
#'   value or range imposed on assortativity coefficients other than
#'   \code{which.range}. It will be ignored if the network is undirected.
#' @param control A list of parameters passed to \code{CVXR::solve()} for
#'   solving an appropriate \code{eta} with the constraints 
#'   \code{target.assortcoef}.
#'
#' @return Range of the interested assortativity coefficient and solver results.
#'
#' @export
#'
#' @examples
#' \donttest{
#' set.seed(123)
#' edgelist <- rpanet(5e3, control =
#'         rpa_control_scenario(alpha = 0.5, beta = 0.5))$edgelist
#' ret1 <- dprewire.range(edgelist, directed = TRUE, which.range = "outin",
#'         target.assortcoef = list("outout" = c(-0.3, 0.3), "inout" = 0.1))
#' ret1$range
#' }
#' 
dprewire.range <- function(edgelist = NULL, directed = TRUE, adj = NULL,
                           which.range = c("outout", "outin", "inout", "inin"),
                           control = cvxr_control(),
                           target.assortcoef = list("outout" = NULL,
                                                    "outin" = NULL,
                                                    "inout" = NULL,
                                                    "inin" = NULL)) {
  if (is.null(edgelist)) {
    if (is.null(adj)) {
      stop('"edgelist" and "adj" can not both be NULL.')
    }
    stopifnot('"dprewire.range" only works for unweighted networks.' = 
                all(adj %% 1 == 0))
    if (! all(adj == 1)) {
      warning('The elements of "adj" are used as the number of edges between nodes.')
    }
    temp <- adj_to_edge(adj = adj, directed = directed, weighted = NULL)
    edgelist <- temp$edgelist
    rm(temp)
  }
  
  stopifnot("Nodes must be consecutive integers starting from 1." = 
            min(edgelist) == 1 & max(edgelist) == length(unique(c(edgelist))))
  
  if (directed) {
    which.range <- match.arg(which.range)
    result <- get_eta_directed(edgelist = edgelist, 
                               target.assortcoef = target.assortcoef,
                               which.range = which.range,
                               control = control)
  }
  else {
    result <- get_eta_undirected(edgelist = edgelist,
                                 control = control)
  }
  result
}
