% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kinetics.R
\name{vo2_kinetics}
\alias{vo2_kinetics}
\title{VO2 kinetics}
\usage{
vo2_kinetics(
  .data,
  intensity_domain = c("moderate", "heavy", "severe"),
  vo2_column = "VO2",
  protocol_n_transitions,
  protocol_baseline_length,
  protocol_transition_length,
  cleaning_level = 0.95,
  cleaning_baseline_fit,
  fit_level = 0.95,
  fit_bin_average,
  fit_phase_1_length,
  fit_baseline_length,
  fit_transition_length,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{.data}{Data retrieved from \code{read_data()}.}

\item{intensity_domain}{The exercise-intensity domain that the test was performed. Either \emph{moderate}, \emph{heavy}, or \emph{severe}.}

\item{vo2_column}{The name (quoted) of the column containing the absolute oxygen uptake (VO2) data. Default to \code{"VO2"}.}

\item{protocol_n_transitions}{Number of transitions performed.}

\item{protocol_baseline_length}{The length of the baseline (in seconds).}

\item{protocol_transition_length}{The length of the transition (in seconds).}

\item{cleaning_level}{A numeric scalar between 0 and 1 giving the confidence level for the intervals to be calculated during the data cleaning process. Breaths lying outside the prediction bands will be excluded. Default to \code{0.95}.}

\item{cleaning_baseline_fit}{A vector of the same length as the number in \code{protocol_n_transitions}, indicating what kind of fit to perform for each baseline. Either \emph{linear} or \emph{exponential}.}

\item{fit_level}{A numeric scalar between 0 and 1 giving the confidence level for the parameter estimates in the final VO2 kinetics fit. Default to \code{0.95}.}

\item{fit_bin_average}{The bin average to be performed for the final fit.}

\item{fit_phase_1_length}{The length of the phase I that you wish to exclude from the final exponential fit, in seconds. See  \verb{VO2 kinetics} section for more details.}

\item{fit_baseline_length}{The length the baseline to perform the final linear fit, in seconds. See \verb{VO2 kinetics} section for more details.}

\item{fit_transition_length}{The length of the transition to perform the final exponential fit, in seconds. See \verb{VO2 kinetics} section for more details.}

\item{verbose}{A boolean indicating whether messages should be printed in the console. Default to \code{TRUE}.}

\item{...}{Additional arguments passed to \code{perform_kinetics()} when fitting VO2 kinetics in the heavy- or severe-intensity domains. See \code{?perform_kinetics} for more details.}
}
\value{
a \link[tibble:tibble-package]{tibble} containing one row and the nested columns:
\item{data_outliers}{The raw data containing additional columns that identify breaths as outliers.}
\item{plot_outliers}{A \code{patchwork} object to display outliers from every transition.}
\item{data_processed}{The processed data (time-aligned, ensembled-averaged, and bin-averaged).}
\item{data_fitted}{The data containing the time and VO2 columns, as well as the fitted data and its residuals for each data point.}
\item{model}{A \code{nls} object. The model used in the VO2 kinetics fitting.}
\item{model_summary}{The tidied summary of the \code{model}.}
\item{model_residuals}{The residuals of the \code{model}.}
\item{plot_model}{The final plot of the fitted \code{model}.}
\item{plot_residuals}{The residuals plot for the \code{model} diagnostics.}
}
\description{
It performs the whole process of the VO2 kinetics data analysis, which includes:
data cleaning (\code{detect_outliers()}); outliers removal, interpolation, ensemble-averaging transitions and bin-avering final dataset (\code{process_data()}),
and modelling VO2 kinetics (\code{perform_kinetics()}). This function is a general function that will call these separate functions.
You can also call each one of them separately if you want.
}
\details{
The function is a wrapper of smaller functions and has important arguments:
\itemize{
\item \strong{protocol_} = sets arguments related to the protocol used.
\item \strong{cleaning_} = sets arguments related to data cleaning.
\item \strong{fit_} = sets arguments related to VO2 kinetics fitting.
}

The function works like the following sequence:

\strong{\code{vo2_kinetics( )}}:
\itemize{
\item \code{detect_outliers( )} = separates the data into the number of transitions indicated,
and fits each baseline and transition phase indiviudally, retrieving the predictions bands for the level indicated.
Then it recognizes breaths lying outside the prediciton bands and flag them as outliers.
\item \code{plot_outliers( )} = plots each transition identifying outliers.
\item \code{process_data( )} = It removes the outliers detected through \code{detect_outliers()}, interpolates each transition,
ensemble-averages all the transitions into one, performs a bin-average, and normalizes the time column
(time zero will indicate the end of baseline and the start of the transition phase).
\item \code{perform_kinetics( )} = performs the VO2 kinetics fitting based on the \strong{fit_} parameters given.
It also calculates the residuals, and plots the final fit as well as residuals for model diagnostics.
}
}
\section{VO2 kinetics}{
VO2 kinetics, described as the rate of adjustment of the oxidative energy system to an
instantaneous increase in the energy demand, is exponential in nature, and it is described by the
oxygen uptake (VO2) time-constant (\eqn{\tau}VO2) (Murias, Spencer and Paterson (2014); Poole and Jones (2011)).

VO2 kinetics analysis provides understanding of the mechanisms that regulate the rate at which oxidative
phosphorylation adapts to step changes in exercise intensities and ATP requirement. This is usually accomplished
by performing step transitions from a baseline intensity to a higher work rate in either the \strong{moderate-}, \strong{heavy-}, or
\strong{severe-intensity domain} (Murias et al., 2011).

Three distinct phases may be observed in the VO2 response during on-transient exercise:

\strong{Phase I}: also termed as the cardiodynamic phase, it represents the circulatory transit delay
on the VO2 response as a result of the increase in the pulmonary blood flow that does not reflect the increase
in oxygen extraction in the active muscles. The time-window of the Phase I is determined in the \strong{\code{fit_phase_1_length}} argument, which will be internally passed into the \code{perform_kinetics()} function.

\strong{Phase II}: also termed as the primary component, represents the exponential increase in VO2
related to the continued increase in pulmonary and muscle blood flow. The Phase II is described by the time-constant parameter (\eqn{\tau})
in the mono-exponential model (see below), and it is defined as the duration of time (in seconds) for the VO2 response
to increase to 63\% of the required steady-state.

\strong{Phase III}: represents the steady-state phase of the VO2 response
during moderate-intensity exercise.
\subsection{Moderate-intensity domain}{

The on-transient response from baseline to a transition within the \strong{moderate-intensity domain}
is analyzed using a \strong{mono-exponential model}:
\deqn{VO_{2\left(t\right)}=baseline+amplitude\cdot\left(1-e^{^{-\frac{\left(t-TD\right)}{tau}}}\right)}{%
VO2(t) = baseline + amplitude * (-exp(-(t-TD)/\tau))}

where:
\itemize{
\item \code{VO2(t)} = the oxygen uptake at any given time.
\item \code{baseline} = the oxygen uptake associated with the baseline phase.
\item \code{amplitude} = the steady-state increase increase in oxygen uptake above \code{baseline}.
\item \code{TD} = the time delay.
\item \eqn{\tau} = the time constant defined as the duration of time for the oxygen uptake to increase to 63\% of the steady-state increase.
}

The baseline value in the mono-exponential model is a \strong{fixed} value and pre-determined
as the mean of the VO2 response (i.e., linear model with the slope set as zero) during the baseline phase.
The time window of the baseline period is determined in the \strong{\code{fit_baseline_length}} argument, which will be internally passed into the \code{perform_kinetics()} function.

Diverse exercise protocols exist to determine VO2 kinetics in the moderate-intensity domain.
Usually, the protocol consists of multiple transitions (typically 3 or 4) from a baseline exercise-intensity to an exercise-intensity
below the gas exchange threshold (typically the power output associated with 90\% of the gas exchange threshold). Bbaseline and
transition phases are usually performed for 6 minutes each. The reason that 6 minutes is done for each phase is to give enough time for both
to reach a steady-state response:

For example, for each multiple of the time-constant (\eqn{\tau}), VO2 increases by 63\% of the
difference between the previous \eqn{\tau} and the required steady-state.
This means:
\itemize{
\item \code{1} \eqn{\tau} \verb{= 63\%} \eqn{\Delta}.
\item \code{2} \eqn{\tau} \verb{= 86\%} \eqn{\Delta} \verb{[100\% - 63\% = 37\%; (37\% x 63\%) + 63\% = 86\%]}.
\item \code{3} \eqn{\tau} \verb{= 95\%} \eqn{\Delta} \verb{[100\% - 86\% = 14\%; (14\% x 63\%) + 86\% = 95\%]}.
\item \code{4} \eqn{\tau} \verb{= 98\%} \eqn{\Delta} \verb{[100\% - 95\% = 5\%; (5\% x 63\%) + 95\% = 98\%]}.
}

In practical terms, let's imagine that a given participant has a \strong{\eqn{\tau} = 60 seconds}. This means that this person
would need \strong{240 seconds} (\verb{4 x 60}) to reach \strong{steady-state} (98\% of the response) in the \strong{moderate-intensity domain}. This would leave other
120 seconds (2 minutes) of transition, so the protocol of performing 6-min transitions makes sure enough time is given.

Now let's imagine that another person has a \strong{\eqn{\tau} = 20 seconds}. This means that this person
would need \strong{80 seconds} (\verb{4 x 20}) to reach \strong{steady-state} (98\% of the response) in the \strong{moderate-intensity domain}.

Given that there is enough time to reach a VO2 steady-state response with 6 minutes of transition, that means that for the final fit
(when the transitions were cleaned, ensembled-averaged, and bin-averaged) there is no need to include the whole 6 minutes of the transition.
This strategy avoids superfluous sections of the steady‐state data, thus maximizing the quality of the fit during the exercise on‐transient (Bell et al., 2001).
This may be specified through the \strong{\code{fit_transition_length}} argument, which will be internally passed into the \code{perform_kinetics()} function.

As for bin-averages in the final fit, usually the data are averaged into 5-s or 10-s bins, 5-s being the most common (Keir et al., 2014).
This may be specified through the \strong{\code{fit_bin_average}} argument, which will be internally passed into the \code{process_data()} function.
}

\subsection{Heavy- and severe-intensity domains}{

TODO
}
}

\examples{
## get file path from example data
path_example <- system.file("example_cosmed.xlsx", package = "whippr")

## read data
df <- read_data(path = path_example, metabolic_cart = "cosmed", time_column = "t")

## VO2 kinetics analysis
results_kinetics <- vo2_kinetics(
  .data = df,
  intensity_domain = "moderate",
  vo2_column = "VO2",
  protocol_n_transitions = 3,
  protocol_baseline_length = 360,
  protocol_transition_length = 360,
  cleaning_level = 0.95,
  cleaning_baseline_fit = c("linear", "exponential", "exponential"),
  fit_level = 0.95,
  fit_bin_average = 5,
  fit_phase_1_length = 20,
  fit_baseline_length = 120,
  fit_transition_length = 240,
  verbose = TRUE
)
}
\references{
Bell, C., Paterson, D. H., Kowalchuk, J. M., Padilla, J., & Cunningham, D. A. (2001). A comparison of modelling techniques used to characterise oxygen uptake kinetics during the on-transient of exercise. Experimental Physiology, 86(5), 667-676.

Keir, D. A., Murias, J. M., Paterson, D. H., & Kowalchuk, J. M. (2014). Breath‐by‐breath pulmonary O2 uptake kinetics: effect of data processing on confidence in estimating model parameters. Experimental physiology, 99(11), 1511-1522.

Murias, J. M., Spencer, M. D., & Paterson, D. H. (2014). The critical role of O2 provision in the dynamic adjustment of oxidative phosphorylation. Exercise and sport sciences reviews, 42(1), 4-11.

Murias, J. M., Spencer, M. D., Kowalchuk, J. M., & Paterson, D. H. (2011). Influence of phase I duration on phase II VO2 kinetics parameter estimates in older and young adults. American Journal of Physiology-regulatory, integrative and comparative physiology, 301(1), R218-R224.

Poole, D. C., & Jones, A. M. (2011). Oxygen uptake kinetics. Comprehensive Physiology, 2(2), 933-996.
}
