% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/xtune.R
\name{xtune}
\alias{xtune}
\title{Regularized regression incorporating external information}
\usage{
xtune(
  X,
  Y,
  Z = NULL,
  U = NULL,
  family = c("linear", "binary", "multiclass"),
  c = 0.5,
  epsilon = 5,
  sigma.square = NULL,
  message = TRUE,
  control = list()
)
}
\arguments{
\item{X}{Numeric design matrix of explanatory variables (\eqn{n} observations in rows, \eqn{p} predictors in columns). \code{xtune} includes an intercept by default.}

\item{Y}{Outcome vector of dimension \eqn{n}.}

\item{Z}{Numeric information matrix about the predictors (\eqn{p} rows, each corresponding to a predictor in X; \eqn{q} columns of external information about the predictors, such as prior biological importance). If Z is the grouping of predictors, it is best if user codes it as a dummy variable (i.e. each column indicating whether predictors belong to a specific group).}

\item{U}{Covariates to be adjusted in the model (matrix with \eqn{n} observations in rows, \eqn{u} predictors in columns). Covariates are non-penalized in the model.}

\item{family}{The family of the model according to different types of outcomes including "linear", "binary", and "multiclass".}

\item{c}{The elastic-net mixing parameter ranging from 0 to 1. When  \eqn{c} = 1, the model corresponds to Lasso. When \eqn{c} is set to 0, it corresponds to Ridge. For values between 0 and 1 (with a default of 0.5), the model corresponds to the elastic net.}

\item{epsilon}{The parameter controls the boundary of the \code{alpha}. The maximum value that \code{alpha} could achieve equals to epsilon times of alpha max calculated by the pathwise coordinate descent. A larger value of epsilon indicates a stronger shrinkage effect (with a default of 5).}

\item{sigma.square}{A user-supplied noise variance estimate. Typically, this is left unspecified, and the function automatically computes an estimated sigma square values using R package \code{selectiveinference}.}

\item{message}{Generates diagnostic message in model fitting. Default is TRUE.}

\item{control}{Specifies \code{xtune} control object. See \code{\link{xtune.control}} for more details.}
}
\value{
An object with S3 class \code{xtune} containing:
\item{beta.est}{The fitted vector of coefficients.}
\item{penalty.vector}{The estimated penalty vector applied to each regression coefficient. Similar to the \code{penalty.factor} argument in \link{glmnet}.}
\item{lambda}{The estimated \eqn{\lambda} value. Note that the lambda value is calculated to reflect that the fact that penalty factors are internally rescaled to sum to nvars in \link{glmnet}. Similar to the \code{lambda} argument in \link{glmnet}.}
\item{alpha.est}{The estimated second-level coefficient for prior covariate Z. The first value is the intercept of the second-level coefficient.}
\item{n_iter}{Number of iterations used until convergence.}
\item{method}{Same as in argument above}
\item{sigma.square}{The estimated sigma square value using \code{\link{estimateVariance}}, if \code{sigma.square} is left unspecified. When \code{family} equals to "binary" or "multiclass", the \code{sigma.square} equals to NULL.}
\item{family}{same as above}
\item{likelihood.score}{A vector containing the marginal likelihood value of the fitted model at each iteration.}
}
\description{
\code{xtune} uses an Empirical Bayes approach to integrate external information into regularized regression models for both linear and categorical outcomes. It fits models with feature-specific penalty parameters based on external information.
}
\details{
\code{xtune} has two main usages:
\itemize{
\item The basic usage of it is to choose the tuning parameter \eqn{\lambda} in elastic net regression using an
Empirical Bayes approach, as an alternative to the widely-used cross-validation. This is done by calling \code{xtune} without specifying external information matrix Z.

\item More importantly, if an external information Z about the predictors X is provided, \code{xtune} can allow predictor-specific shrinkage
parameters for regression coefficients in penalized regression models. The idea is that Z might be informative for the effect-size of regression coefficients, therefore we can guide the penalized regression model using Z.
}

Please note that the number of rows in Z should match with the number of columns in X. Since each column in Z is a feature about X. \href{https://github.com/JingxuanH/xtune}{See here for more details on how to specify Z}.

A majorization-minimization procedure is employed to fit \code{xtune}.
}
\examples{
## use simulated example data
set.seed(1234567)
data(example)
X <- example$X
Y <- example$Y
Z <- example$Z

## Empirical Bayes tuning to estimate tuning parameter, as an alternative to cross-validation:
\donttest{
fit.eb <- xtune(X=X,Y=Y, family = "linear")
fit.eb$lambda
}

### compare with tuning parameter chosen by cross-validation, using glmnet
\donttest{
fit.cv <- glmnet::cv.glmnet(x=X,y=Y,alpha = 0.5)
fit.cv$lambda.min
}

## Feature-specific penalties based on external information Z:
\donttest{
fit.diff <- xtune(X=X,Y=Y,Z=Z, family = "linear")
fit.diff$penalty.vector
}

}
\seealso{
\link{predict_xtune}, as well as \link{glmnet}.
}
\author{
Jingxuan He and Chubing Zeng
}
