% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fast_variants.R
\name{fast_discom}
\alias{fast_discom}
\title{Fast DISCOM}
\usage{
fast_discom(
  beta,
  x,
  y,
  x.tuning,
  y.tuning,
  x.test,
  y.test,
  nlambda,
  pp,
  robust = 0,
  n.l = 30,
  standardize = TRUE,
  itcp = TRUE,
  lambda.min.ratio = NULL,
  k.value = 1.5
)
}
\arguments{
\item{beta}{Vector, true beta coefficients (optional)}

\item{x}{Matrix, training data}

\item{y}{Vector, training response}

\item{x.tuning}{Matrix, tuning data}

\item{y.tuning}{Vector, tuning response}

\item{x.test}{Matrix, test data}

\item{y.test}{Vector, test response}

\item{nlambda}{Integer, number of lambda values}

\item{pp}{Vector, block sizes}

\item{robust}{Integer, 0 for classical, 1 for robust estimation}

\item{n.l}{Integer, number of tuning parameter (`l`) values for fast variants}

\item{standardize}{Logical, whether to standardize covariates. When TRUE, uses training data mean and standard deviation to standardize tuning and test sets. When robust=1, uses Huber-robust standard deviation estimates}

\item{itcp}{Logical, whether to include intercept}

\item{lambda.min.ratio}{Numeric, `lambda.min.ratio` sets the smallest lambda value in the grid, expressed as a fraction of `lambda.max`—the smallest lambda for which all coefficients are zero. By default, it is `0.0001` when the number of observations (`nobs`) exceeds the number of variables (`nvars`), and `0.01` when `nobs < nvars`. Using a very small value in the latter case can lead to overfitting.}

\item{k.value}{Numeric, tuning parameter for robust estimation}
}
\value{
List with estimation results
}
\description{
Fast DISCOM
}
\section{Value}{

The function returns a list containing the following components:

\describe{
  \item{err}{A multi-dimensional array storing the mean squared error (MSE) for all combinations of tuning parameters alpha and lambda.}
  \item{est.error}{The estimation error, calculated as the Euclidean distance between the estimated beta coefficients and the true beta (if provided).}
  \item{lambda}{The optimal lambda value chosen via cross-validation on the tuning set.}
  \item{alpha}{A vector of the optimal alpha values, also selected on the tuning set.}
  \item{train.error}{The mean squared error on the tuning set for the optimal parameter combination.}
  \item{test.error}{The mean squared error on the test set for the final, optimal model.}
  \item{y.pred}{The predicted values for the observations in the test set.}
  \item{R2}{The R-squared value, which measures the proportion of variance explained by the model on the test set.}
  \item{a0}{The intercept of the final model.}
  \item{a1}{The vector of estimated beta coefficients for the final model.}
  \item{select}{The number of non-zero coefficients, representing the number of selected variables.}
  \item{xtx}{The final regularized covariance matrix used to fit the optimal model.}
  \item{fpr}{The False Positive Rate (FPR) if the true beta is provided. It measures the proportion of irrelevant variables incorrectly selected.}
  \item{fnr}{The False Negative Rate (FNR) if the true beta is provided. It measures the proportion of relevant variables incorrectly excluded.}
  \item{lambda.all}{The complete vector of all lambda values tested during cross-validation.}
  \item{beta.cov.lambda.max}{The estimated beta coefficients using the maximum lambda value.}
  \item{time}{The total execution time of the function in seconds.}
}
}

\examples{
\donttest{
# Fast DISCOM example with synthetic multimodal data
n <- 70
p <- 18

# Generate synthetic data with 3 blocks
set.seed(321)
x_train <- matrix(rnorm(n * p), n, p)
x_tuning <- matrix(rnorm(35 * p), 35, p)
x_test <- matrix(rnorm(20 * p), 20, p)

# True coefficients with block structure
beta_true <- c(rep(1.0, 3), rep(0, 3), rep(-1.2, 3), rep(0, 3), rep(0.8, 3), rep(0, 3))

# Response variables
y_train <- x_train \%*\% beta_true + rnorm(n, sd = 0.4)
y_tuning <- x_tuning \%*\% beta_true + rnorm(35, sd = 0.4)
y_test <- x_test \%*\% beta_true + rnorm(20, sd = 0.4)

# Block sizes (3 blocks of 6 variables each)
pp <- c(6, 6, 6)

# Run fast DISCOM (efficient for large datasets)
result <- fast_discom(beta = beta_true,
                      x = x_train, y = y_train,
                      x.tuning = x_tuning, y.tuning = y_tuning,
                      x.test = x_test, y.test = y_test,
                      nlambda = 20, pp = pp, n.l = 25)

# View results
print(paste("Test error:", round(result$test.error, 4)))
print(paste("R-squared:", round(result$R2, 3)))
print(paste("Runtime:", round(result$time, 2), "seconds"))
}
}
