% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/AncReg.R
\name{AncReg}
\alias{AncReg}
\title{Ancestor Regression}
\usage{
AncReg(x, degree = 0, targets = colnames(x), f = function(x) x^3)
}
\arguments{
\item{x}{A named numeric matrix containing the observational data.}

\item{degree}{An integer specifying the order of the SVAR process to be considered. Default is 0 for no time series.}

\item{targets}{A character vector specifying the variables whose ancestors should be estimated. Default is all variables.}

\item{f}{A function specifying the non-linearity used in the ancestor regression. Default is a cubic function.}
}
\value{
An object of class "AncReg" containing:
\item{z.val}{A numeric matrix of test statistics.}
\item{p.val}{A numeric matrix of p-values.}
}
\description{
This function performs ancestor regression for linear structural equation models \insertCite{schultheiss2024ancestorregressionstructuralvector}{AncReg} and
vector autoregressive models \insertCite{ancestor}{AncReg} with explicit error control for false discovery, at least asymptomatically.
}
\examples{
##### simulated example for inear structural equation models

# random DAGS for simulation
set.seed(1234)

p <- 5 #number of nodes
DAG <- pcalg::randomDAG(p, prob = 0.5)

B <- matrix(0, p, p) # represent DAG as matrix
for (i in 2:p){
  for(j in 1:(i-1)){
    # store edge weights
    B[i,j] <- max(0, DAG@edgeData@data[[paste(j,"|",i, sep="")]]$weight)
  }
}
colnames(B) <- rownames(B) <- LETTERS[1:p]

# solution in terms of noise
Bprime <- MASS::ginv(diag(p) - B)

n <- 5000
N <- matrix(rexp(n * p), ncol = p)
X <- t(Bprime \%*\% t(N))
colnames(X) <- LETTERS[1:p]

# fit ancestor regression
fit <- AncReg(X)
# collect ancestral p-values and graph
res <- summary(fit)
res

#compare true and estimated ancestral graph
trueGraph <- igraph::graph_from_adjacency_matrix(recAncestor(B != 0))
ancGraph <- igraph::graph_from_adjacency_matrix(res$graph)

oldpar <- par(mfrow = c(1, 2))
plot(trueGraph, main = 'true ancestral graph', vertex.size = 30)
plot(ancGraph, main = 'Ancestor Regression', vertex.size = 30)

##### SVAR-example with geyser timeseries
geyser <- MASS::geyser
# shift waiting such that it is waiting after erruption
geyser2 <- data.frame(waiting = geyser$waiting[-1], duration = geyser$duration[-nrow(geyser)])

# fit ancestor regression with 6 lags considered
fit2 <- AncReg(as.matrix(geyser2), degree = 6)
res2 <- summary(fit2)
res2

# visualize instantaneous ancestry
instGraph <- igraph::graph_from_adjacency_matrix(res2$inst.graph)
plot(instGraph, edge.label = round(diag(res2$inst.p.val[1:2, 2:1]), 2),
     main = 'instantaneous effects', vertex.size = 90)

# visualize summary of lagged ancestry
sumGraph <- igraph::graph_from_adjacency_matrix(res2$sum.graph)
plot(sumGraph, edge.label = round(diag(res2$sum.p.val[1:2, 2:1]), 2),
     main = 'summary graph', vertex.size = 90)
par(oldpar)
}
\references{
\insertAllCited{}
}
\seealso{
\code{\link{summary.AncReg}}, \code{\link{instant_graph}}, \code{\link{summary_graph}},
\code{\link{instant_p.val}}, \code{\link{summary_p.val}}
}
