% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ggplot_rsi.R
\name{ggplot_rsi}
\alias{ggplot_rsi}
\alias{geom_rsi}
\alias{facet_rsi}
\alias{scale_y_percent}
\alias{scale_rsi_colours}
\alias{theme_rsi}
\title{AMR bar plots with \code{ggplot}}
\usage{
ggplot_rsi(data, position = "stack", x = "Antibiotic",
  fill = "Interpretation", facet = NULL, translate_ab = "official",
  ...)

geom_rsi(position = "stack", x = c("Antibiotic", "Interpretation"),
  fill = "Interpretation", translate_ab = "official")

facet_rsi(facet = c("Interpretation", "Antibiotic"), ...)

scale_y_percent()

scale_rsi_colours()

theme_rsi()
}
\arguments{
\item{data}{a \code{data.frame} with column(s) of class \code{"rsi"} (see \code{\link{as.rsi}})}

\item{position}{position adjustment of bars, either \code{"stack"} (default) or \code{"dodge"}}

\item{x}{variable to show on x axis, either \code{"Antibiotic"} (default) or \code{"Interpretation"} or a grouping variable}

\item{fill}{variable to categorise using the plots legend, either \code{"Antibiotic"} (default) or \code{"Interpretation"} or a grouping variable}

\item{facet}{variable to split plots by, either \code{"Interpretation"} (default) or \code{"Antibiotic"} or a grouping variable}

\item{translate_ab}{a column name of the \code{\link{antibiotics}} data set to translate the antibiotic abbreviations into, using \code{\link{abname}}. Default behaviour is to translate to official names according to the WHO. Use \code{translate_ab = FALSE} to disable translation.}

\item{...}{other parameters passed on to \code{\link[ggplot2]{facet_wrap}}}
}
\description{
Use these functions to create bar plots for antimicrobial resistance analysis. All functions rely on internal \code{\link[ggplot2]{ggplot}} functions.
}
\details{
At default, the names of antibiotics will be shown on the plots using \code{\link{abname}}. This can be set with the option \code{get_antibiotic_names} (a logical value), so change it e.g. to \code{FALSE} with \code{options(get_antibiotic_names = FALSE)}.

\strong{The functions}\cr
\code{geom_rsi} will take any variable from the data that has an \code{rsi} class (created with \code{\link{as.rsi}}) using \code{\link{portion_df}} and will plot bars with the percentage R, I and S. The default behaviour is to have the bars stacked and to have the different antibiotics on the x axis.

\code{facet_rsi} creates 2d plots (at default based on S/I/R) using \code{\link[ggplot2]{facet_wrap}}.

\code{scale_y_percent} transforms the y axis to a 0 to 100\% range.

\code{scale_rsi_colours} sets colours to the bars: green for S, yellow for I and red for R.

\code{theme_rsi} is a \code{\link[ggplot2]{theme}} with minimal distraction.

\code{ggplot_rsi} is a wrapper around all above functions that uses data as first input. This makes it possible to use this function after a pipe (\code{\%>\%}). See Examples.
}
\examples{
library(dplyr)
library(ggplot2)

# get antimicrobial results for drugs against a UTI:
ggplot(septic_patients \%>\% select(amox, nitr, fosf, trim, cipr)) +
  geom_rsi()

# prettify the plot using some additional functions:
df <- septic_patients[, c("amox", "nitr", "fosf", "trim", "cipr")]
ggplot(df) +
  geom_rsi() +
  facet_rsi() +
  scale_y_percent() +
  scale_rsi_colours() +
  theme_rsi()

# or better yet, simplify this using the wrapper function - a single command:
septic_patients \%>\%
  select(amox, nitr, fosf, trim, cipr) \%>\%
  ggplot_rsi()
\donttest{
# it also supports groups (don't forget to use the group on `x` or `facet`):
septic_patients \%>\%
  select(hospital_id, amox, nitr, fosf, trim, cipr) \%>\%
  group_by(hospital_id) \%>\%
  ggplot_rsi(x = "hospital_id",
             facet = "Antibiotic",
             nrow = 1) +
  labs(title = "AMR of Anti-UTI Drugs Per Hospital",
       x = "Hospital")

# genuine analysis: check 2 most prevalent microorganisms
septic_patients \%>\%
  # create new bacterial ID's, with all CoNS under the same group (Becker et al.)
  mutate(bactid = as.bactid(bactid, Becker = TRUE)) \%>\%
  # filter on top 2 bacterial ID's
  filter(bactid \%in\% top_freq(freq(.$bactid), 2)) \%>\%
  # determine first isolates
  mutate(first_isolate = first_isolate(.,
                                       col_date = "date",
                                       col_patient_id = "patient_id",
                                       col_bactid = "bactid")) \%>\%
  # filter on first isolates
  filter(first_isolate == TRUE) \%>\%
  # join the `microorganisms` data set
  left_join_microorganisms() \%>\%
  # select full name and some antiseptic drugs
  select(mo = fullname,
            cfur, gent, cipr) \%>\%
  # group by MO
  group_by(mo) \%>\%
  # plot the thing, putting MOs on the facet
  ggplot_rsi(x = "Antibiotic",
             facet = "mo") +
  labs(title = "AMR of Top Two Microorganisms In Blood Culture Isolates",
       subtitle = "Only First Isolates, CoNS grouped according to Becker et al.",
       x = "Microorganisms")
}
}
