% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/count.R
\name{count}
\alias{count}
\alias{count_R}
\alias{count_IR}
\alias{count_I}
\alias{count_SI}
\alias{count_S}
\alias{count_all}
\alias{n_rsi}
\alias{count_df}
\title{Count isolates}
\source{
Wickham H. \strong{Tidy Data.} The Journal of Statistical Software, vol. 59, 2014. \url{http://vita.had.co.nz/papers/tidy-data.html}
}
\usage{
count_R(..., only_all_tested = FALSE)

count_IR(..., only_all_tested = FALSE)

count_I(..., only_all_tested = FALSE)

count_SI(..., only_all_tested = FALSE)

count_S(..., only_all_tested = FALSE)

count_all(..., only_all_tested = FALSE)

n_rsi(..., only_all_tested = FALSE)

count_df(data, translate_ab = "name", language = get_locale(),
  combine_SI = TRUE, combine_IR = FALSE)
}
\arguments{
\item{...}{one or more vectors (or columns) with antibiotic interpretations. They will be transformed internally with \code{\link{as.rsi}} if needed.}

\item{only_all_tested}{(for combination therapies, i.e. using more than one variable for \code{...}) a logical to indicate that isolates must be tested for all antibiotics, see section \emph{Combination therapy} below}

\item{data}{a \code{data.frame} containing columns with class \code{rsi} (see \code{\link{as.rsi}})}

\item{translate_ab}{a column name of the \code{\link{antibiotics}} data set to translate the antibiotic abbreviations to, using \code{\link{ab_property}}}

\item{language}{language of the returned text, defaults to system language (see \code{\link{get_locale}}) and can also be set with \code{\link{getOption}("AMR_locale")}. Use \code{language = NULL} or \code{language = ""} to prevent translation.}

\item{combine_SI}{a logical to indicate whether all values of S and I must be merged into one, so the output only consists of S+I vs. R (susceptible vs. resistant). This used to be the parameter \code{combine_IR}, but this now follows the redefinition by EUCAST about the interpretion of I (increased exposure) in 2019, see section 'Interpretation of S, I and R' below. Default is \code{TRUE}.}

\item{combine_IR}{a logical to indicate whether all values of I and R must be merged into one, so the output only consists of S vs. I+R (susceptible vs. non-susceptible). This is outdated, see parameter \code{combine_SI}.}
}
\value{
Integer
}
\description{
These functions can be used to count resistant/susceptible microbial isolates. All functions support quasiquotation with pipes, can be used in \code{dplyr}s \code{\link[dplyr]{summarise}} and support grouped variables, see \emph{Examples}.

\code{count_R} and \code{count_IR} can be used to count resistant isolates, \code{count_S} and \code{count_SI} can be used to count susceptible isolates.\cr
}
\details{
These functions are meant to count isolates. Use the \code{\link{portion}_*} functions to calculate microbial resistance.

The function \code{n_rsi} is an alias of \code{count_all}. They can be used to count all available isolates, i.e. where all input antibiotics have an available result (S, I or R). Their use is equal to \code{\link{n_distinct}}. Their function is equal to \code{count_S(...) + count_IR(...)}.

The function \code{count_df} takes any variable from \code{data} that has an \code{"rsi"} class (created with \code{\link{as.rsi}}) and counts the amounts of S, I and R. The resulting \emph{tidy data} (see Source) \code{data.frame} will have three rows (S/I/R) and a column for each variable with class \code{"rsi"}.

The function \code{rsi_df} works exactly like \code{count_df}, but adds the percentage of S, I and R.
}
\section{Interpretation of S, I and R}{

In 2019, EUCAST has decided to change the definitions of susceptibility testing categories S, I and R as shown below (\url{http://www.eucast.org/newsiandr/}). Results of several consultations on the new definitions are available on the EUCAST website under "Consultations".

\itemize{
  \item{\strong{S} - }{Susceptible, standard dosing regimen: A microorganism is categorised as "Susceptible, standard dosing regimen", when there is a high likelihood of therapeutic success using a standard dosing regimen of the agent.}
  \item{\strong{I} - }{Susceptible, increased exposure: A microorganism is categorised as "Susceptible, Increased exposure" when there is a high likelihood of therapeutic success because exposure to the agent is increased by adjusting the dosing regimen or by its concentration at the site of infection.}
  \item{\strong{R} - }{Resistant: A microorganism is categorised as "Resistant" when there is a high likelihood of therapeutic failure even when there is increased exposure.}
}

Exposure is a function of how the mode of administration, dose, dosing interval, infusion time, as well as distribution and excretion of the antimicrobial agent will influence the infecting organism at the site of infection.

This AMR package honours this new insight. Use \code{\link{portion_SI}} to determine antimicrobial susceptibility and \code{\link{count_SI}} to count susceptible isolates.
}

\section{Combination therapy}{

When using more than one variable for \code{...} (= combination therapy)), use \code{only_all_tested} to only count isolates that are tested for all antibiotics/variables that you test them for. See this example for two antibiotics, Antibiotic A and Antibiotic B, about how \code{portion_SI} works to calculate the \%SI:

\preformatted{
--------------------------------------------------------------------
                    only_all_tested = FALSE  only_all_tested = TRUE
                    -----------------------  -----------------------
 Drug A    Drug B   include as  include as   include as  include as
                    numerator   denominator  numerator   denominator
--------  --------  ----------  -----------  ----------  -----------
 S or I    S or I       X            X            X            X
   R       S or I       X            X            X            X
  <NA>     S or I       X            X            -            -
 S or I      R          X            X            X            X
   R         R          -            X            -            X
  <NA>       R          -            -            -            -
 S or I     <NA>        X            X            -            -
   R        <NA>        -            -            -            -
  <NA>      <NA>        -            -            -            -
--------------------------------------------------------------------
}

Please note that, in combination therapies, for \code{only_all_tested = TRUE} applies that:
\preformatted{
   count_S()  +  count_I()  +  count_R()  == count_all()
  portion_S() + portion_I() + portion_R() == 1
}
and that, in combination therapies, for \code{only_all_tested = FALSE} applies that:
\preformatted{
   count_S()  +  count_I()  +  count_R()  >= count_all()
  portion_S() + portion_I() + portion_R() >= 1
}

Using \code{only_all_tested} has no impact when only using one antibiotic as input.
}

\section{Read more on our website!}{

On our website \url{https://msberends.gitlab.io/AMR} you can find \href{https://msberends.gitlab.io/AMR/articles/AMR.html}{a tutorial} about how to conduct AMR analysis, the \href{https://msberends.gitlab.io/AMR/reference}{complete documentation of all functions} (which reads a lot easier than here in R) and \href{https://msberends.gitlab.io/AMR/articles/WHONET.html}{an example analysis using WHONET data}.
}

\examples{
# example_isolates is a data set available in the AMR package.
?example_isolates

# Count resistant isolates
count_R(example_isolates$AMX)
count_IR(example_isolates$AMX)

# Or susceptible isolates
count_S(example_isolates$AMX)
count_SI(example_isolates$AMX)

# Count all available isolates
count_all(example_isolates$AMX)
n_rsi(example_isolates$AMX)

# Since n_rsi counts available isolates, you can
# calculate back to count e.g. non-susceptible isolates.
# This results in the same:
count_SI(example_isolates$AMX)
portion_SI(example_isolates$AMX) * n_rsi(example_isolates$AMX)

library(dplyr)
example_isolates \%>\%
  group_by(hospital_id) \%>\%
  summarise(R  = count_R(CIP),
            I  = count_I(CIP),
            S  = count_S(CIP),
            n1 = count_all(CIP),  # the actual total; sum of all three
            n2 = n_rsi(CIP),      # same - analogous to n_distinct
            total = n())          # NOT the number of tested isolates!

# Count co-resistance between amoxicillin/clav acid and gentamicin,
# so we can see that combination therapy does a lot more than mono therapy.
# Please mind that `portion_SI` calculates percentages right away instead.
count_SI(example_isolates$AMC)  # 1433
count_all(example_isolates$AMC) # 1879

count_SI(example_isolates$GEN)  # 1399
count_all(example_isolates$GEN) # 1855

with(example_isolates,
     count_SI(AMC, GEN))        # 1764
with(example_isolates,
     n_rsi(AMC, GEN))           # 1936

# Get portions S/I/R immediately of all rsi columns
example_isolates \%>\%
  select(AMX, CIP) \%>\%
  count_df(translate = FALSE)

# It also supports grouping variables
example_isolates \%>\%
  select(hospital_id, AMX, CIP) \%>\%
  group_by(hospital_id) \%>\%
  count_df(translate = FALSE)

}
\seealso{
\code{\link{portion}_*} to calculate microbial resistance and susceptibility.
}
\keyword{antibiotics}
\keyword{isolate}
\keyword{isolates}
\keyword{resistance}
\keyword{rsi}
\keyword{susceptibility}
