#####################################################
#             Importing a CSV file                  #
#####################################################
#' Importing a CSV file
#'
#' Import a CSV file containing the output of the MCMC algorithm
#'
#' Use of the \code{read.csv()} function with default values for CSV files produced by
#' \href{https://chronomodel.com}{ChronoModel} software. For MCMC in a date format
#' different from BC/AD, use the
#' parameter \code{referenceYear} to convert the MCMC to BC/AD, otherwise the
#' remaining functions of \pkg{ArchaeoPhases} will not work. MCMC files generated by
#' \href{https://bcal.shef.ac.uk/top.html}{BCal} may contain an empty last row. This row
#' should be withdrawn using the \code{rowToWithdraw} parameter.
#' Otherwise, the functions of \pkg{ArchaeoPhases} will not work properly.
#'
#' @param file Name of the CSV file containing the output of the MCMC algorithm.
#' @param dec Character used in the file for decimal points for the use of \code{read.csv()}.
#' @param sep Field separator character for the use of \code{read.csv()}.
#' @param comment.char Character vector of length one containing a single
#' character or an empty string for the use of \code{read.csv()}.
#' @param header Logical value indicating whether the file contains the names
#' of the variables as its first line.
#' @param iterationColumn Column number corresponding to the iteration values,
#' default = \code{NULL}.
#' @param referenceYear Year of reference for MCMC in date format other than
#' BC/AD, default = \code{NULL}.
#' @param rowToWithdraw Number of the row to be withdrawn or "last" for the
#' last row of the data frame, default = \code{NULL}.
#' @param bin.width Bin width specified in a BCal project (note that \code{bin.width}
#' does not have to be set if the BCal default bin width of 1 is used).
#'
#' @return A data frame containing a representation of the data in the file.
#'
#' @author Anne Philippe, \email{Anne.Philippe@@univ-nantes.fr},
#' @author Thomas S. Dye, \email{tsd@@tsdye.online}, and
#' @author Marie-Anne Vibet, \email{Marie-Anne.Vibet@@univ-nantes.fr}
#'
#' @examples
#'
#'   data(Events)
#' \dontrun{
#'   write.csv(Events, "data.csv", row.names=FALSE)
#'   data = ImportCSV("data.csv", dec = '.', sep=',', comment.char='#',
#'                    header = TRUE, iterationColumn = 1)
#'
#'   # Import of MCMC generated by BCal and extracted in cal BP
#'   # (the year of reference is 1950)
#'   if (requireNamespace("ArchaeoPhases.dataset", quietly = TRUE)) {
#'   data(Fishpond)
#'   write.csv(Fishpond, "fishpond_MCMC.csv", row.names=FALSE)
#'   Fishpond = ImportCSV("fishpond_MCMC.csv", dec = '.', sep=',',
#'                        header = TRUE, iterationColumn = 1,
#'                        referenceYear = 1950, rowToWithdraw = "last")}
#' }
#'
#'
#' @seealso \code{\link{ImportCSV.BCal}}
#' @seealso \code{\link{read_chronomodel}}
#' @seealso \code{\link{read_oxcal}}
#'
#' @importFrom utils read.csv
#'
#' @export
#'
ImportCSV <- function(file, dec='.', sep=',', comment.char = '#',
                      header = TRUE, iterationColumn = NULL,
                      referenceYear = NULL, rowToWithdraw = NULL,
                      bin.width = NULL)
{
  # importing the CSV file
  data = read.csv(file, dec = dec, sep=sep, comment.char = comment.char,
                  header = header)

  # Withdrawing the iterations column
  if (!is.null(iterationColumn)){
    data = data[,-iterationColumn]
  }

  # Withdrawing a row
  if (!is.null(rowToWithdraw)){
    if (is.numeric(rowToWithdraw)) {
      data = data[-rowToWithdraw, ]
    }else{
      if (rowToWithdraw == "last") {
        data = data[-nrow(data), ]
      }
    }
  }

  # BCal bin width
  if (!is.null(bin.width)) {
    L = length(data)
    unbin <- function(value, width) {
      value * width
    }
    for (i in 1:L) {
      if (is.numeric(data[, i]) == TRUE) {
        data[, i] = sapply(data[, i], unbin, bin.width)
      }

    }
  }

  # Conversion of the MCMC samples in date format cal BP or other to BC/AD
  if (!is.null(referenceYear)){
    data2 = data
    L = length(data)
    conv <- function(value, T0){
      T0 - value
    }
    for (i in 1:L){
      if( is.numeric(data[,i]) == TRUE){
        data2[,i] = sapply(data[,i], conv, referenceYear)
      }
    }

    data = data2
  }

  return(data)

}

#' Read MCMC output from OxCal
#'
#' Import a CSV file containing the output of the MCMC algorithm produced
#' by \href{https://c14.arch.ox.ac.uk/oxcal.html}{OxCal}.
#'
#' The \code{read_oxcal} function is built on \code{\link[readr]{read_csv}}.
#' It aims to be fast and simple, and to return the marginal posteriors free
#' of extraneous artifacts.  The iteration column in the CSV file is discarded,
#' as is an empty last column.
#'
#' @param file Either a path to a CSV file, a connection,
#' or the value \code{clipboard()} to read from the system clipboard.
#' The CSV file can be compressed or plain.
#' See \code{\link[readr]{read_csv}} for details.
#'
#' @param quiet One of "no" (default) to allow messages and warnings, "partial"
#' to suppress messages and allow warnings, or "yes" to suppress messages
#' and warnings.
#'
#' @return An \code{archaeophases_mcmc} object containing the marginal
#' posterior(s) as a data frame.
#'
#' @author Thomas S. Dye, \email{tsd@@tsdye.online}
#'
#' @examples
#'
#' \dontrun{
#'   # Import of MCMC output from OxCal
#'   data(Events)
#'   #To do for saving in csv file
#'   # write.csv(Events, "events.csv", row.names = FALSE)
#'   fishpond <- read_oxcal("events.csv")
#'
#'   # Read from connection
#'   oxc <- read_oxcal("http://tsdye.online/AP/ox.csv")
#' }
#'
#' @seealso \code{\link[readr]{read_csv}}
#' @seealso \code{\link{ImportCSV}}
#'
#' @importFrom readr read_csv
#' @importFrom dplyr %>% summarise_all
#' @importFrom digest digest
#' @importFrom utils file_test
#'
#' @export
read_oxcal <- function(file, quiet="no")
{
    ## OxCal hard codes csv file conventions, per C. Bronk Ramsey
    ## These match the R defaults
    data <- switch(quiet,
                   "no" = read_csv(file),
                   "partial" = suppressMessages(read_csv(file)),
                   "yes" = suppressWarnings(suppressMessages(read_csv(file))),
                   read_csv(file))
    ## Calculate hash, if connection, save temp file
    if(!file_test("-f", file)) {
        temp_file <- tempfile(pattern = "", fileext = "csv")
        write(file, temp_file)
        file_hash <- digest(file = temp_file, algo = "sha256")
        unlink(temp_file)
   }
    else {
        file_hash <- digest(file = file, algo = "sha256")
    }
    ## Remove the iteration column
    data <- data[, -1]
    ## OxCal uses trailing commas in MCMC output, so trim the last column,
    ## which is empty
    if (data[, ncol(data)] %>% summarise_all(class) != "numeric")
        data <- data[, -ncol(data)]
    ## Return archaeophases_mcmc object
    new_archaeophases_mcmc(x = as.data.frame(data),
                           call = match.call(),
                           hash = file_hash)
}

#' Read MCMC output from ChronoModel
#'
#' Import a CSV file containing the output of the MCMC algorithm produced
#' by \href{https://chronomodel.com/}{ChronoModel}.
#'
#' The \code{read_chronomodel} function is built on
#' \code{\link[readr]{read_delim}}.
#' It aims to be fast and simple, and to return the marginal posteriors
#' free of extraneous artifacts.  The iteration column in the CSV file is
#' discarded.
#'
#' @param file Either a path to a CSV file, a connection,
#' or the value \code{clipboard()} to read from the system clipboard.
#' The CSV file can be compressed or plain.
#' See \code{\link[readr]{read_delim}} for details.
#' @param decimal Either "." (default) or ",",
#' the two choices offered by
#' \href{https://chronomodel.com/}{ChronoModel}.
#' @param separator The character used to separate fields
#' in the CSV file.  Defaults to ",".
#' @param quiet One of "no" (default) to allow messages and warnings, "partial"
#' to suppress messages and allow warnings, or "yes" to suppress messages
#' and warnings.
#'
#' @return An \code{archaeophases_mcmc} object containing the marginal
#' posterior(s) from file.
#'
#' @author Thomas S. Dye, \email{tsd@@tsdye.online}
#'
#' @examples
#'
#'   data(Events)
#' \dontrun{
#'   write.csv(Events, "events.csv", row.names=FALSE)
#'   events = read_chronomodel("events.csv", decimal = ".", separator = ",")
#'   # equivalent
#'   events = read_chronomodel("events.csv")
#'
#'
#'    rem <- read_chronomodel("http://tsdye.online/AP/cm/Chain_all_Events.csv")
#'  }
#'
#' @seealso \code{\link[readr]{read_delim}}
#' @seealso \code{\link{ImportCSV}}
#' @seealso \code{\link{new_archaeophases_mcmc}}
#'
#' @importFrom readr read_delim locale
#' @importFrom digest digest
#' @importFrom utils file_test
#'
#' @export
read_chronomodel <- function(file, decimal = ".", separator = ",", quiet = "no")
{
    ## ChronoModel allows the user to choose any separator
    ## and either a period or comma for decimals
    data <- switch(quiet,
                   "no" = read_delim(file,
                                     locale = locale("en", decimal_mark = decimal),
                                     delim = separator, comment = "#"),
                   "partial" = suppressMessages(
                       read_delim(file,
                                  locale = locale("en", decimal_mark = decimal),
                                  delim = separator, comment = "#")),
                   "yes" = suppressWarnings(
                       suppressMessages(read_delim(file,
                                                   locale = locale("en", decimal_mark = decimal),
                                                   delim = separator, comment = "#"))),
                   read_delim(file,
                              locale = locale("en", decimal_mark = decimal),
                              delim = separator, comment = "#"))
    ## Calculate hash, if connection, save temp file
    if(!file_test("-f", file)) {
        temp_file <- tempfile(pattern = "", fileext = "csv")
        write(file, temp_file)
        file_hash <- digest(file = temp_file, algo = "sha256")
        unlink(temp_file)
    }
    else {
        file_hash <- digest(file = file, algo = "sha256")
    }
    ## Remove the iteration column
    data <- data[, -1]
    ## Return archaeophases_mcmc object
    new_archaeophases_mcmc(x = as.data.frame(data),
                           call = match.call(),
                           hash = file_hash)
}

#' Read MCMC output from BCal
#'
#' Import a CSV file containing the output of the MCMC algorithm produced
#' by \href{https://bcal.shef.ac.uk/}{BCal}.
#'
#' The \code{read_bcal} function is built on \code{\link[readr]{read_csv}}.
#' It aims to be fast and simple, and to return the marginal posteriors free
#' of extraneous artifacts.  The iteration column in the CSV file is discarded,
#' as are an empty last column and an empty last row.
#'
#' @param file Either a path to a CSV file, a connection,
#' or the value \code{clipboard()} to read from the system clipboard.
#' The CSV file can be compressed or plain.
#' See \code{\link[readr]{read_csv}} for details.
#' @param bin_width The bin width specified for the
#' \href{https://bcal.shef.ac.uk/}{BCal} calibration.
#' Defaults to the \href{https://bcal.shef.ac.uk/}{BCal} default of 1.
#' @param quiet One of "no" (default) to allow messages and warnings, "partial"
#' to suppress messages and allow warnings, or "yes" to suppress messages
#' and warnings.
#'
#' @return An \code{archaeophases_mcmc} object containing the marginal
#' posterior(s) as a data frame.
#'
#' @author Thomas S. Dye, \email{tsd@@tsdye.online}
#'
#' @examples
#'
#' \dontrun{
#'   # Import of MCMC output from BCal
#'   data(Fishpond)
#'   write.csv(Fishpond, "fishpond_MCMC.csv", row.names=FALSE)
#'   fishpond <- read_bcal("fishpond_MCMC.csv")
#'
#' # Read from connection
#'   bc_1 <- read_bcal("http://tsdye.online/AP/bc-1.csv")
#'   bc_17 <- read_bcal("http://tsdye.online/AP/bc-17.csv", bin_width = 17)
#'}
#'
#' @seealso \code{\link[readr]{read_csv}}
#' @seealso \code{\link{ImportCSV}}
#' @seealso \code{\link{new_archaeophases_mcmc}}
#'
#' @importFrom readr read_csv
#' @importFrom dplyr %>% summarise_all
#' @importFrom digest digest
#' @importFrom utils file_test
#'
#' @export
read_bcal <- function(file, bin_width = 1, quiet = "no")
{
    ## BCal uses English locale csv conventions
    data <- switch(quiet,
                   "no" = read_csv(file),
                   "partial" = suppressMessages(read_csv(file)),
                   "yes" = suppressMessages(suppressWarnings(read_csv(file))),
                   read_csv(file))
    ## Calculate hash, if connection, save temp file
    if(!file_test("-f", file)) {
        temp_file <- tempfile(pattern = "", fileext = "csv")
        write(file, temp_file)
        file_hash <- digest(file = temp_file, algo = "sha256")
        unlink(temp_file)
    }
    else {
        file_hash <- digest(file = file, algo = "sha256")
    }
    ## Remove the iteration column
    data <- data[, -1]
    ## Remove an empty last column, if present
    if (data[, ncol(data)] %>% summarise_all(class) != "numeric")
        data <- data[, -ncol(data)]
    ## BCal used to add an empty row at the end, check if empty and remove
    suppressWarnings(if (is.na(data[nrow(data), ]))
        data <- data[-nrow(data), ])
    ## Take bin width into account, if necessary
    if (bin_width != 1)
        data <- bin_width * data
    ## Convert from BP to BC/AD
    data <- 1950 - data
    ## Return archaeophases_mcmc object
    new_archaeophases_mcmc(x = as.data.frame(data),
                           call = match.call(),
                           hash = file_hash)
}
