% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pkgname.R
\docType{package}
\name{BIOMASS-package}
\alias{BIOMASS}
\alias{BIOMASS-package}
\title{BIOMASS: Estimating Aboveground Biomass and Its Uncertainty in Tropical Forests}
\description{
Contains functions to estimate aboveground biomass/carbon and its uncertainty in tropical forests. 
	These functions allow to (1) retrieve and to correct taxonomy, (2) estimate wood density and its uncertainty, 
	(3) construct height-diameter models, (4) manage tree and plot coordinates, 
	(5) estimate the aboveground biomass/carbon at the stand level with associated uncertainty. 
	To cite BIOMASS, please use citation("BIOMASS"). 
	See more in the article of Réjou-Méchain et al. (2017) <doi:10.1111/2041-210X.12753>.
}
\examples{
\dontrun{
library(BIOMASS)

# Dataset containing plot inventory data from Karnataka, India (Ramesh et al. 2010)
data(KarnatakaForest)
str(KarnatakaForest)

# Dataset containing height and diameter measurements from two 1-ha plots
# established in the lowland rainforest of French Guiana, at the Nouragues
# Ecological Research Station
data(NouraguesHD)
str(NouraguesHD)

#############################################################################
# WOOD DENSITY

# 1-RETRIEVE AND CORRECT TAXONOMY

# Checking typos in taxonomy
Taxo <- correctTaxo(genus = KarnatakaForest$genus, species = KarnatakaForest$species)
KarnatakaForest$genusCorr <- Taxo$genusCorrected
KarnatakaForest$speciesCorr <- Taxo$speciesCorrected

# Retrieving APG III Families and Orders from Genus names
APG <- getTaxonomy(KarnatakaForest$genusCorr, findOrder = T)
KarnatakaForest$familyAPG <- APG$family
KarnatakaForest$orderAPG <- APG$order

# 2-RETRIEVE WOOD DENSITY
dataWD <- getWoodDensity(
  genus = KarnatakaForest$genusCorr,
  species = KarnatakaForest$speciesCorr,
  stand = KarnatakaForest$plotID
)

#############################################################################
# TREE HEIGHT

# Compare different local H-D models
modelHD(
  D = NouraguesHD$D, H = NouraguesHD$H,
  drawGraph = TRUE, useWeight = TRUE
)

# Compute the local H-D model with the lowest RSE
HDmodel <- modelHD(
  D = NouraguesHD$D, H = NouraguesHD$H,
  method = "log2", useWeight = TRUE
)

# Compute plot-specific H-D models
HDmodelPerPlot <- modelHD(NouraguesHD$D, NouraguesHD$H,
  method = "weibull",
  useWeight = T, plot = NouraguesHD$plotId
)

RSEmodels <- sapply(HDmodelPerPlot, function(x) x$RSE)
Coeffmodels <- lapply(HDmodelPerPlot, function(x) x$coefficients)

# Retrieve height data from a local HD model
dataHlocal <- retrieveH(D = KarnatakaForest$D, model = HDmodel)

# Retrieve height data from a Feldpaush et al. (2012) averaged model
dataHfeld <- retrieveH(D = KarnatakaForest$D, region = "SEAsia")

# Retrieve height data from Chave et al. (2012) equation 6
dataHchave <- retrieveH(
  D = KarnatakaForest$D,
  coord = cbind(KarnatakaForest$long, KarnatakaForest$lat)
)

#############################################################################
# AGB CALCULATION

KarnatakaForest$WD <- dataWD$meanWD
KarnatakaForest$H <- dataHlocal$H
KarnatakaForest$Hfeld <- dataHfeld$H

# Compute AGB(Mg) per tree
AGBtree <- computeAGB(
  D = KarnatakaForest$D, WD = KarnatakaForest$WD,
  H = KarnatakaForest$H
)

# Compute AGB(Mg) per plot
AGBplot <- summaryByPlot(AGBtree, KarnatakaForest$plotId)

# Compute AGB(Mg) per tree without height information (Eq. 7 from Chave et al. (2014))
AGBplotChave <- summaryByPlot(
  computeAGB(
    D = KarnatakaForest$D, WD = KarnatakaForest$WD,
    coord = KarnatakaForest[, c("long", "lat")]
  ),
  plot = KarnatakaForest$plotId
)

# Compute AGB(Mg) per tree with Feldpausch et al. (2012) regional H-D model
AGBplotFeld <- summaryByPlot(
  computeAGB(
    D = KarnatakaForest$D, WD = KarnatakaForest$WD,
    H = KarnatakaForest$Hfeld
  ),
  plot = KarnatakaForest$plotId
)

#############################################################################
# PROPAGATING ERRORS

KarnatakaForest$sdWD <- dataWD$sdWD
KarnatakaForest$HfeldRSE <- dataHfeld$RSE

# Per plot using the local HD model constructed above (modelHD)
resultMC <- AGBmonteCarlo(
  D = KarnatakaForest$D, WD = KarnatakaForest$WD, errWD = KarnatakaForest$sdWD,
  HDmodel = HDmodel, Dpropag = "chave2004"
)
resMC <- summaryByPlot(resultMC$AGB_simu, KarnatakaForest$plotId)

# Per plot using the Feldpaush regional HD averaged model
AGBmonteCarlo(
  D = KarnatakaForest$D, WD = KarnatakaForest$WD,
  errWD = KarnatakaForest$sdWD, H = KarnatakaForest$Hfeld,
  errH = KarnatakaForest$HfeldRSE, Dpropag = "chave2004"
)
resMC <- summaryByPlot(resultMC$AGB_simu, KarnatakaForest$plotId)

# Per plot using Chave et al. (2014) Equation 7
resultMC <- AGBmonteCarlo(
  D = KarnatakaForest$D, WD = KarnatakaForest$WD, errWD = KarnatakaForest$sdWD,
  coord = KarnatakaForest[, c("long", "lat")],
  Dpropag = "chave2004"
)
resMC <- summaryByPlot(resultMC$AGB_simu, KarnatakaForest$plotId)
}
}
\references{
Réjou-Méchain M., Tanguy A., Piponiot C., Chave J., Hérault B. (2017). BIOMASS :
An R Package for estimating above-ground biomass and its uncertainty in tropical forests.
Methods in Ecology and Evolution, 8(9), 1163-1167.
}
\seealso{
Useful links:
\itemize{
  \item \url{https://github.com/AMAP-dev/BIOMASS}
  \item Report bugs at \url{https://github.com/AMAP-dev/BIOMASS/issues}
}

}
\author{
\strong{Maintainer}: Maxime Réjou-Méchain \email{maxime.rejou@gmail.com} [data contributor]

Authors:
\itemize{
  \item Arthur Pere \email{arthur.pere@cirad.fr}
  \item Guillaume Cornu \email{gcornu@cirad.fr} (0000-0002-7523-5176)
  \item Ariane Tanguy
  \item Camille Piponiot
  \item Bruno Hérault \email{bruno.herault@cirad.fr}
}

Other contributors:
\itemize{
  \item Jerome Chave \email{jerome.chave@univ-tlse3.fr} [data contributor]
  \item Ted Feldpausch \email{T.R.Feldpausch@exeter.ac.uk} [data contributor]
  \item Philippe Verley \email{philippe.verley@ird.fr } [contributor]
}

}
\keyword{internal}
