% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/BootstrapQTL.R
\name{BootstrapQTL}
\alias{BootstrapQTL}
\title{Bootstrap QTL analysis for accurate effect size estimation}
\usage{
BootstrapQTL(snps, gene, snpspos, genepos, cvrt = SlicedData$new(),
  n_bootstraps = 200, n_cores = 1, eGene_detection_file_name = NULL,
  bootstrap_file_directory = NULL, cisDist = 1e+06,
  local_correction = "bonferroni", global_correction = "fdr",
  correction_type = "shrinkage", errorCovariance = numeric(),
  useModel = modelLINEAR, eigenMT_tests_per_gene = NULL)
}
\arguments{
\item{snps}{\code{\link[MatrixEQTL:SlicedData-class]{SlicedData}} object containing genotype
information used as input into \code{\link[MatrixEQTL]{Matrix_eQTL_main}}.}

\item{gene}{\code{\link[MatrixEQTL:SlicedData-class]{SlicedData}} object containing gene expression
information used as input into \code{\link[MatrixEQTL]{Matrix_eQTL_main}}.}

\item{snpspos}{\code{data.frame} object with information about SNP locations.
Used in conjunction with \code{'genespos'} and \code{'cisDist'} to
determine SNPs in \emph{cis} of each gene. Must have three columns: \enumerate{
  \item 'snpid' describing the name of the SNP and corresponding to rows in
  the 'snps' matrix.
  \item 'chr' describing the chromosome for each SNP.
  \item 'pos' describing the position of the SNP on the chromosome.
}}

\item{genepos}{\code{data.frame} object with information about transcript locations.
Used in conjunction with \code{'snpspos'} and \code{'cisDist'} to
determine SNPs in \emph{cis} of each gene. Must have four columns: \enumerate{
  \item 'geneid' describing the name of the gene and corresponding to rows in
  the 'gene' matrix.
  \item 'chr' describing the chromosome for each SNP.
  \item 'left' describing the start position of the transcript.
  \item 'right' describing the end position of the transcript.
}
Note that \code{\link[MatrixEQTL]{Matrix_eQTL_main}} tests all
variants within \code{cisDist} of the start or end of the gene.
If you wish instead to test all variants within \code{cisDist} of
the transcription start site, you should specify this location in
both the 'left' and 'right' columns of the \code{genepos}
data.frame. Similarly, when analysing a molecular phenotype that
have a single chromosomal position then the 'left' and 'right'
columns should both contain the same position.}

\item{cvrt}{\code{\link[MatrixEQTL:SlicedData-class]{SlicedData}} object containing covariate
information used as input into \code{\link[MatrixEQTL]{Matrix_eQTL_main}}.
Argument can be ignored in the case of no covariates.}

\item{n_bootstraps}{number of bootstraps to run.}

\item{n_cores}{number of cores to parallise the bootstrap procedure
over.}

\item{eGene_detection_file_name}{\code{character}, \code{connection} or \code{NULL}.
File to save local \emph{cis} associations to in the eGene detection analysis. Corresponds
to the \code{output_file_name.cis} argument in \code{\link[MatrixEQTL]{Matrix_eQTL_main}}.
If a file with this name exists it is overwritten, if \code{NULL} output is not saved
to file.}

\item{bootstrap_file_directory}{\code{character} or \code{NULL}. If not \code{NULL},
files will be saved in this directory storing local \emph{cis} associations for
the bootstrap eGene detection group (detection_bootstrapnumber.txt) and local
\emph{cis} associations the bootstrap left-out eGene effect size estimation
group (estimation_bootstrapnumber.txt). Estimation group files will only be saved
where signficant eGenes are also significant in the bootstrap detection group
(see Details). Corresponds to the \code{output_file_name.cis} argument in the
respective calls to \code{\link[MatrixEQTL]{Matrix_eQTL_main}}. Files in this
directory will be overwritten if they already exist.}

\item{cisDist}{\code{numeric}. Argument to \code{\link[MatrixEQTL]{Matrix_eQTL_main}}
controlling the maximum distance from a gene to consider tests for
eQTL mapping.}

\item{local_correction}{multiple testing correction method to use when
correcting p-values across all SNPs at each gene (see EQTL mapping
section in Details). Can be a method specified in \code{\link[stats:p.adjust]{p.adjust.methods}},
"qvalue" for the \code{\link[qvalue]{qvalue}} package, or "eigenMT"
if EigenMT has been used to estimate the number effective independent 
tests (see \code{eigenMT_tests_per_gene}).}

\item{global_correction}{multiple testing correction method to use when
correcting p-values across all genes after performing local correction
(see EQTL mapping section in Details). Must be a method specified in
\code{\link[stats:p.adjust]{p.adjust.methods}} or "qvalue" for the
\code{\link[qvalue]{qvalue}} package.}

\item{correction_type}{\code{character}. One of "shrinkage", "out_of_sample"
or "weighted". Determines which Winner's Curse correction method is
used (see Details).}

\item{errorCovariance}{\code{numeric matrix} argument to \code{\link[MatrixEQTL]{Matrix_eQTL_main}}
specifying the error covariance.}

\item{useModel}{\code{integer} argument to \code{\link[MatrixEQTL]{Matrix_eQTL_main}}
specifying the type of model to fit between each SNP and gene. Should be one of
\code{\link[MatrixEQTL]{modelLINEAR}}, \code{\link[MatrixEQTL]{modelANOVA}}, or
\code{\link[MatrixEQTL]{modelLINEAR_CROSS}}.}

\item{eigenMT_tests_per_gene}{\code{data.frame} containing the number of effective
independent tests for each gene estimated by the EigenMT (\url{https://github.com/joed3/eigenMT}).
Ignore unless \code{'local_correction="eigenMT"'}.}
}
\value{
A \code{data.frame} (or \code{\link[data.table]{data.table}} if the
 user has the library loaded) containing the results for each significant eQTL:
 \describe{
   \item{\code{'eGene':}}{The eQTL eGene.}
   \item{\code{'eSNP':}}{The eQTL eSNP.}
   \item{\code{'statistic':}}{The test statistic for the association between the eGene and eSNP.}
   \item{\code{'nominal_beta':}}{The eQTL effect size for the \code{eGene}-\code{eSNP}
     pair estimated by \code{\link[MatrixEQTL:MatrixEQTL-package]{MatrixEQTL}}.}
   \item{\code{'corrected_beta':}}{The eQTL effect size after adjustment for the \code{winners_curse}.}
   \item{\code{'winners_curse':}}{The amount of effect size overestimation determined by the
     bootstrap analysis (See Details).}
   \item{\code{'correction_boots':}}{The number of bootstraps that contributed to the estimation of
     the \code{winners_curse}, \emph{i.e.} the number of bootstraps in which the \code{eSNP}
     remained significantly associated with the \code{eGene} (see Details).}
   \item{\code{'nominal_pval':}}{The p-value for the \code{eGene}-\code{eSNP} pair
     from the \code{\link[MatrixEQTL:MatrixEQTL-package]{MatrixEQTL}} analysis.}
   \item{\code{'eSNP_pval':}}{The locally corrected p-value for the \code{eGene}-\code{eSNP} pair (see Details).}
   \item{\code{'eGene_pval':}}{The globally corrected p-value for the \code{eGene} based on its top eSNP (see Details).}
 }
}
\description{
Performs cis-QTL mapping using MatrixEQTL then performs a bootstrap
analysis to obtain unbiased effect size estimates for traits with
significant evidence of genetic regulation correcting for the
"Winner's Curse" arising from lead-SNP selection.
}
\details{
Although the package interface and documentation describe the use of
 \code{BootstrapQTL} for \emph{cis}-eQTL mapping, the package can be
 applied to any QTL study of quantitative traits with chromosomal
 positions, for example \emph{cis}-QTL mapping of epigenetic
 modifications. Any matrix of molecular trait data can be provided
 to the \code{'gene'} argument provided a corresponding \code{'genepos'}
 'data.frame' detailing the chromosomal positions of each trait is
 provided.

 \subsection{Cis-eQTL mapping:}{
 EQTL mapping is performed using the
 \code{\link[MatrixEQTL:MatrixEQTL-package]{MatrixEQTL}} package. A three step
 hieararchical multiple testing correction procedure is used to
 determine significant eGenes and eSNPs. At the first step, nominal
 p-values from \code{\link[MatrixEQTL:MatrixEQTL-package]{MatrixEQTL}} for all
 \emph{cis}-SNPs are adjusted for each gene separately using the
 method specified in the \code{'local_correction'} argument
 (Bonferroni correction by default). In the second step, the best
 adjusted p-value is taken for each gene, and this set of locally
 adjusted p-values is corrected for multiple testing across all genes
 using the methods pecified in the \code{'global_correction'} argument
 (FDR correction by default). In the third step, an eSNP significance
 threshold on the locally corrected p-values is determined as the
 locally corrected p-value corresponding to the globally corrected
 p-value threshold of 0.05.

 A gene is considered a significant eGene if its globally corrected
 p-value is < 0.05, and a SNP is considered a significant eSNP for
 that eGene if its locally corrected p-value < the eSNP significance
 threshold.

 The default settings for \code{'local_correction'} and
 \code{'global_correction'} were found to best control eGene false
 discovery rate without sacrificing sensitivity (see citation).
 }
 \subsection{Winner's Curse correction:}{
 EQTL effect sizes of significant eSNPs on significant eGenes are
 typically overestimated when compared to replication datasets
 (see citation). \code{BootstrapEQTL} removes this overestimation by
 performing a bootstrap procedure after eQTL mapping.

 Three Winner's Curse correction methods are available: the Shrinkage
 method, the Out of Sample method, and the Weighted Estimator method.
 All three methods work on the same basic principle of performing
 repeated sample bootstrapping to partition the dataset into two
 groups: an eQTL detection group comprising study samples select via
 random sampling with replacement, and an eQTL effect size estimation
 group comprising the remaining samples not selected via the random
 sampling. The default estimator, \code{'correction_type = "shrinkage"'},
 provided the most accurate corrected effect sizes in our simulation
 study (see citation).

 The \strong{shrinkage method} ("shrinkage" in
 \code{'correction_type'}) corrects for the Winner's Curse by
 measuring the average difference between the eQTL effect size
 in the bootstrap detection group and the bootstrap estimation group,
 then subtracting this difference from the naive eQTL effect size
 estimate obtained from the eGene detection analysis prior to the
 bootstrap procedure.

 The \strong{out of sample method} ("out_of_sample" in
 \code{'correction_type'}) corrects for the Winner's Curse by taking
 the average eQTL effect size across bootstrap estimation groups as
 an unbiased effect size estimate.

 The \strong{weighted estimator method} ("weighted" in
 \code{'correction_type'}) corrects for the Winner's Curse by taking
 a weighted average of the nominal estimate of the eQTL effect size
 and the average of eQTL effect sizes across the bootstrap estimation
 groups: \eqn{0.368 * naive_estimate + 0.632 *
 mean(bootstrap estimation group effect sizes)}.

 In all three methods bootstrap effect sizes only contribute to
 the Winner's Curse correction if the corresponding eSNP is
 significantly associated with the eGene in the bootstrap detection
 group (locally corrected bootstrap P-value < eSNP significance
 threshold determing in the eQTL mapping step).

 Note that eQTLs may not remain significant in all bootstraps, so the
 effective number of bootstraps used to obtain the Winner's Curse
 estimate will typically be lower than the number of bootstraps
 specified in \code{'n_bootstraps'}. The number of bootstraps that
 were significant for each eQTL are reported in the
 \code{'correction_boots'} column of the returned table.
 }
 \subsection{Winner's Curse corrected effect sizes}{
 The user should be aware that ability to correct for Winner's Curse
 can vary across significant eQTLs depending on their statistical
 power (\emph{i.e. minor allele frequency, true effect size, and
 study sample size}). Users should be skeptical of corrected effect
 sizes that are larger than the nominal effect sizes estimated by
 \code{\link[MatrixEQTL:MatrixEQTL-package]{MatrixEQTL}}, which likely reflects low
 power for eQTL detection rather than an underestimated effect size.
 }
 \subsection{Bootstrap warning messages:}{
 It is possible for bootstrap analyses to fail due to the reduced
 sample sizes of the bootstrap detection and bootstrap estimation
 groups. For example, the bootstrap resampling may lead to an
 detection or estimation groups in which all individuals are
 homozygous for an eSNP or have no variance in their supplied
 covariates (\emph{e.g.} the estimation group may comprise
 individuals all of the same sex). In this case the bootstrap will
 fail for all eQTLs since \code{\link[MatrixEQTL:MatrixEQTL-package]{MatrixEQTL}} will
 be unable to perform the model fitting.

 Failed bootstraps are reported after the bootstrap procedure in
 a series of warning messages indicating the number of bootstrap
 failures grouped by the reason for the bootstrap failure.
 }
}
\examples{
# Locations for example data from the MatrixEQTL package
base.dir = find.package('MatrixEQTL');
SNP_file_name = paste(base.dir, "/data/SNP.txt", sep="");
snps_location_file_name = paste(base.dir, "/data/snpsloc.txt", sep="");
expression_file_name = paste(base.dir, "/data/GE.txt", sep="");
gene_location_file_name = paste(base.dir, "/data/geneloc.txt", sep="");
covariates_file_name = paste(base.dir, "/data/Covariates.txt", sep="");

# Load the SNP data
snps = SlicedData$new();
snps$fileDelimiter = "\\t";      # the TAB character
snps$fileOmitCharacters = "NA"; # denote missing values;
snps$fileSkipRows = 1;          # one row of column labels
snps$fileSkipColumns = 1;       # one column of row labels
snps$fileSliceSize = 2000;      # read file in slices of 2,000 rows
snps$LoadFile(SNP_file_name);

# Load the data detailing the position of each SNP
snpspos = read.table(snps_location_file_name, header = TRUE, stringsAsFactors = FALSE);

# Load the gene expression data
gene = SlicedData$new();
gene$fileDelimiter = "\\t";      # the TAB character
gene$fileOmitCharacters = "NA"; # denote missing values;
gene$fileSkipRows = 1;          # one row of column labels
gene$fileSkipColumns = 1;       # one column of row labels
gene$fileSliceSize = 2000;      # read file in slices of 2,000 rows
gene$LoadFile(expression_file_name);

# Load the data detailing the position of each gene
genepos = read.table(gene_location_file_name, header = TRUE, stringsAsFactors = FALSE);

# Load the covariates data
cvrt = SlicedData$new();
cvrt$fileDelimiter = "\\t";      # the TAB character
cvrt$fileOmitCharacters = "NA"; # denote missing values;
cvrt$fileSkipRows = 1;          # one row of column labels
cvrt$fileSkipColumns = 1;       # one column of row labels
if(length(covariates_file_name)>0) {
  cvrt$LoadFile(covariates_file_name);
}

# Run the BootstrapQTL analysis
eQTLs <- BootstrapQTL(snps, gene, snpspos, genepos, cvrt, n_bootstraps=10, n_cores=2)

}
